"""
Kontext提示词小秘 - 工具函数
"""

import json
import re
from typing import Dict, Any, List, Optional


def validate_api_url(url: str) -> bool:
    """
    验证API URL格式
    """
    url_pattern = re.compile(
        r'^https?://'  # http:// or https://
        r'(?:(?:[A-Z0-9](?:[A-Z0-9-]{0,61}[A-Z0-9])?\.)+[A-Z]{2,6}\.?|'  # domain...
        r'localhost|'  # localhost...
        r'\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})'  # ...or ip
        r'(?::\d+)?'  # optional port
        r'(?:/?|[/?]\S+)$', re.IGNORECASE)
    return url_pattern.match(url) is not None


def validate_json_string(json_str: str) -> tuple[bool, Optional[Dict[str, Any]]]:
    """
    验证JSON字符串格式
    
    Returns:
        tuple: (是否有效, 解析后的字典或None)
    """
    try:
        parsed = json.loads(json_str)
        if isinstance(parsed, dict):
            return True, parsed
        else:
            return False, None
    except json.JSONDecodeError:
        return False, None


def format_template_variables(template: str, variables: Dict[str, Any]) -> str:
    """
    格式化模板变量
    
    Args:
        template: 模板字符串
        variables: 变量字典
    
    Returns:
        格式化后的字符串
    """
    try:
        return template.format(**variables)
    except KeyError as e:
        missing_var = str(e).strip("'\"")
        return f"模板变量错误: 缺少变量 '{missing_var}'\n\n原始模板:\n{template}"
    except Exception as e:
        return f"模板格式化错误: {str(e)}\n\n原始模板:\n{template}"


def extract_template_variables(template: str) -> List[str]:
    """
    从模板中提取所有变量名
    
    Args:
        template: 模板字符串
    
    Returns:
        变量名列表
    """
    pattern = r'\{([^}]+)\}'
    matches = re.findall(pattern, template)
    return list(set(matches))  # 去重


def sanitize_api_response(response_text: str) -> str:
    """
    清理API响应文本
    
    Args:
        response_text: 原始响应文本
    
    Returns:
        清理后的文本
    """
    if not response_text:
        return ""
    
    # 移除可能的控制字符
    cleaned = re.sub(r'[\x00-\x08\x0b\x0c\x0e-\x1f\x7f-\x9f]', '', response_text)
    
    # 标准化换行符
    cleaned = re.sub(r'\r\n|\r', '\n', cleaned)
    
    # 移除首尾空白
    cleaned = cleaned.strip()
    
    return cleaned


def build_error_message(error_type: str, details: str, suggestions: List[str] = None) -> str:
    """
    构建格式化的错误消息
    
    Args:
        error_type: 错误类型
        details: 错误详情
        suggestions: 建议解决方案列表
    
    Returns:
        格式化的错误消息
    """
    message = f"错误类型: {error_type}\n详情: {details}"
    
    if suggestions:
        message += "\n\n建议解决方案:"
        for i, suggestion in enumerate(suggestions, 1):
            message += f"\n{i}. {suggestion}"
    
    return message


def estimate_token_count(text: str) -> int:
    """
    估算文本的token数量（粗略估算）
    
    Args:
        text: 输入文本
    
    Returns:
        估算的token数量
    """
    # 简单的token估算：英文按单词计算，中文按字符计算
    chinese_chars = len(re.findall(r'[\u4e00-\u9fff]', text))
    english_words = len(re.findall(r'\b[a-zA-Z]+\b', text))
    other_chars = len(text) - chinese_chars - sum(len(word) for word in re.findall(r'\b[a-zA-Z]+\b', text))
    
    # 粗略估算：中文1字符≈1token，英文1单词≈1.3token，其他字符按0.5计算
    estimated_tokens = chinese_chars + int(english_words * 1.3) + int(other_chars * 0.5)
    
    return max(1, estimated_tokens)


def create_api_headers(api_key: str, additional_headers: Dict[str, str] = None) -> Dict[str, str]:
    """
    创建API请求头
    
    Args:
        api_key: API密钥
        additional_headers: 额外的请求头
    
    Returns:
        完整的请求头字典
    """
    headers = {
        'Content-Type': 'application/json',
        'Authorization': f'Bearer {api_key}',
        'User-Agent': 'Kontext-ComfyUI-Plugin/1.0'
    }
    
    if additional_headers:
        headers.update(additional_headers)
    
    return headers


def parse_model_response(response_data: Dict[str, Any]) -> tuple[str, Dict[str, Any]]:
    """
    解析模型响应数据
    
    Args:
        response_data: API响应数据
    
    Returns:
        tuple: (提取的文本内容, 元数据信息)
    """
    try:
        # 标准OpenAI格式
        if 'choices' in response_data and len(response_data['choices']) > 0:
            choice = response_data['choices'][0]
            
            # Chat Completions格式
            if 'message' in choice:
                content = choice['message'].get('content', '')
            # Completions格式
            elif 'text' in choice:
                content = choice['text']
            else:
                content = str(choice)
            
            # 提取元数据
            metadata = {
                'model': response_data.get('model', ''),
                'usage': response_data.get('usage', {}),
                'finish_reason': choice.get('finish_reason', ''),
                'created': response_data.get('created', 0)
            }
            
            return content, metadata
        
        # 其他格式的响应
        elif 'text' in response_data:
            return response_data['text'], {}
        elif 'content' in response_data:
            return response_data['content'], {}
        else:
            return str(response_data), {}
    
    except Exception as e:
        return f"响应解析错误: {str(e)}", {}


class PromptBuilder:
    """
    提示词构建器
    """
    
    def __init__(self):
        self.system_parts = []
        self.user_parts = []
    
    def add_system_instruction(self, instruction: str) -> 'PromptBuilder':
        """添加系统指令"""
        if instruction.strip():
            self.system_parts.append(instruction.strip())
        return self
    
    def add_user_input(self, input_text: str) -> 'PromptBuilder':
        """添加用户输入"""
        if input_text.strip():
            self.user_parts.append(input_text.strip())
        return self
    
    def add_context(self, context: str, context_type: str = "背景信息") -> 'PromptBuilder':
        """添加上下文信息"""
        if context.strip():
            formatted_context = f"{context_type}:\n{context.strip()}"
            self.user_parts.append(formatted_context)
        return self
    
    def add_examples(self, examples: List[Dict[str, str]]) -> 'PromptBuilder':
        """添加示例"""
        if examples:
            example_text = "示例:\n"
            for i, example in enumerate(examples, 1):
                example_text += f"\n示例{i}:\n"
                example_text += f"输入: {example.get('input', '')}\n"
                example_text += f"输出: {example.get('output', '')}\n"
            self.user_parts.append(example_text.strip())
        return self
    
    def build(self) -> tuple[str, str]:
        """
        构建最终的提示词
        
        Returns:
            tuple: (系统提示词, 用户提示词)
        """
        system_prompt = "\n\n".join(self.system_parts) if self.system_parts else ""
        user_prompt = "\n\n".join(self.user_parts) if self.user_parts else ""
        
        return system_prompt, user_prompt
    
    def clear(self) -> 'PromptBuilder':
        """清空所有内容"""
        self.system_parts.clear()
        self.user_parts.clear()
        return self