#!/usr/bin/env python3
"""
Kontext提示词小秘 - 测试脚本
用于测试节点功能的独立脚本
"""

import sys
import os
import json

# 添加当前目录到Python路径
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

from nodes import KontextPromptAssistant, KontextPromptTemplate


def test_prompt_template():
    """测试提示词模板节点"""
    print("=" * 50)
    print("测试提示词模板节点")
    print("=" * 50)
    
    template_node = KontextPromptTemplate()
    
    # 测试创意写作模板
    variables = json.dumps({
        "topic": "人工智能的未来发展",
        "style": "科普文章",
        "length": "800字"
    }, ensure_ascii=False)
    
    result = template_node.generate_template(
        template_type="创意写作",
        custom_template="",
        variables=variables
    )
    
    print("模板类型: 创意写作")
    print("变量:", variables)
    print("\n生成的模板:")
    print("-" * 30)
    print(result[0])
    print("-" * 30)
    
    # 测试自定义模板
    custom_template = "请以{style}的方式，写一篇关于{topic}的{type}，字数控制在{length}以内。"
    custom_variables = json.dumps({
        "style": "幽默风趣",
        "topic": "程序员的日常生活",
        "type": "短文",
        "length": "500字"
    }, ensure_ascii=False)
    
    result2 = template_node.generate_template(
        template_type="自定义",
        custom_template=custom_template,
        variables=custom_variables
    )
    
    print("\n\n自定义模板测试:")
    print("原始模板:", custom_template)
    print("变量:", custom_variables)
    print("\n生成的模板:")
    print("-" * 30)
    print(result2[0])
    print("-" * 30)


def test_prompt_assistant_mock():
    """测试提示词助手节点（模拟模式，不实际调用API）"""
    print("\n\n" + "=" * 50)
    print("测试提示词助手节点（模拟模式）")
    print("=" * 50)
    
    assistant_node = KontextPromptAssistant()
    
    # 显示输入类型信息
    input_types = assistant_node.INPUT_TYPES()
    print("节点输入参数:")
    for category, params in input_types.items():
        print(f"\n{category.upper()}:")
        for param_name, param_config in params.items():
            if isinstance(param_config, tuple):
                param_type, config = param_config
                default = config.get('default', 'N/A')
                print(f"  - {param_name}: {param_type} (默认: {default})")
            else:
                print(f"  - {param_name}: {param_config}")
    
    print(f"\n返回类型: {assistant_node.RETURN_TYPES}")
    print(f"返回名称: {assistant_node.RETURN_NAMES}")
    print(f"函数名称: {assistant_node.FUNCTION}")
    print(f"分类: {assistant_node.CATEGORY}")
    
    # 测试参数验证（使用无效的API密钥）
    print("\n测试参数验证:")
    result = assistant_node.generate_response(
        api_base_url="https://api.openai.com/v1",
        api_key="",  # 空API密钥
        model="gpt-3.5-turbo",
        system_prompt="你是一个有用的AI助手。",
        user_input="你好，请介绍一下自己。",
        temperature=0.7,
        max_tokens=100
    )
    
    print(f"生成文本: '{result[0]}'")
    print(f"完整响应: '{result[1]}'")
    print(f"状态信息: '{result[2]}'")


def test_integration():
    """测试节点集成"""
    print("\n\n" + "=" * 50)
    print("测试节点集成")
    print("=" * 50)
    
    # 创建模板
    template_node = KontextPromptTemplate()
    variables = json.dumps({
        "topic": "ComfyUI插件开发",
        "style": "技术教程",
        "length": "300字"
    }, ensure_ascii=False)
    
    template_result = template_node.generate_template(
        template_type="创意写作",
        custom_template="",
        variables=variables
    )
    
    print("步骤1: 生成模板")
    print("生成的系统提示词:")
    print("-" * 30)
    print(template_result[0])
    print("-" * 30)
    
    # 模拟使用模板结果作为系统提示词
    assistant_node = KontextPromptAssistant()
    print("\n步骤2: 使用模板作为系统提示词（模拟）")
    print("如果有有效的API密钥，将会调用:")
    print(f"  系统提示词: {template_result[0][:100]}...")
    print(f"  用户输入: 请详细介绍ComfyUI节点开发的基本步骤")


def main():
    """主测试函数"""
    print("Kontext提示词小秘 - 节点测试")
    print("测试时间:", __import__('datetime').datetime.now().strftime("%Y-%m-%d %H:%M:%S"))
    
    try:
        test_prompt_template()
        test_prompt_assistant_mock()
        test_integration()
        
        print("\n\n" + "=" * 50)
        print("✅ 所有测试完成！")
        print("=" * 50)
        print("\n注意事项:")
        print("1. 这是离线测试，未实际调用API")
        print("2. 要测试API功能，请在ComfyUI中使用有效的API密钥")
        print("3. 确保安装了requirements.txt中的依赖")
        
    except Exception as e:
        print(f"\n❌ 测试过程中出现错误: {e}")
        import traceback
        traceback.print_exc()


if __name__ == "__main__":
    main()