#!/usr/bin/env python3
"""
DeepSeek API 测试脚本
测试你提供的API配置是否正常工作
"""

import sys
import os
import json

# 添加当前目录到Python路径
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

from nodes import KontextPromptAssistant


def test_deepseek_api():
    """测试DeepSeek API配置"""
    print("=" * 60)
    print("DeepSeek R1 API 测试")
    print("=" * 60)
    
    # 你的API配置
    api_config = {
        "api_base_url": "https://api.ds.lobeapi.com/v1",
        "api_key": "sk-YdPWWj1gQGwgqs95uKtu0AnkWj2OyNp37zhDRf0gyFG2wwVI",
        "model": "deepseek-r1",
        "temperature": 0.7,
        "max_tokens": 500,
        "timeout": 30,
        "retry_count": 3
    }
    
    print("API配置信息:")
    print(f"  Base URL: {api_config['api_base_url']}")
    print(f"  Model: {api_config['model']}")
    print(f"  API Key: {api_config['api_key'][:20]}...{api_config['api_key'][-10:]}")
    print(f"  Temperature: {api_config['temperature']}")
    print(f"  Max Tokens: {api_config['max_tokens']}")
    
    # 创建节点实例
    assistant_node = KontextPromptAssistant()
    
    # 测试用例
    test_cases = [
        {
            "name": "基础对话测试",
            "system_prompt": "你是DeepSeek开发的AI助手，擅长逻辑推理和问题解决。请简洁明了地回答问题。",
            "user_input": "请简单介绍一下你自己，包括你的能力特点。"
        },
        {
            "name": "逻辑推理测试", 
            "system_prompt": "你是一位逻辑推理专家。请分析问题并给出清晰的推理过程。",
            "user_input": "如果所有的猫都是动物，而小花是一只猫，那么小花是什么？请解释推理过程。"
        },
        {
            "name": "创意写作测试",
            "system_prompt": "你是一位富有创意的作家。请根据给定的元素创作一个简短的故事片段。",
            "user_input": "关键词：机器人、图书馆、神秘的书。请写一个100字左右的故事开头。"
        }
    ]
    
    # 执行测试
    for i, test_case in enumerate(test_cases, 1):
        print(f"\n{'='*20} 测试 {i}: {test_case['name']} {'='*20}")
        print(f"系统提示词: {test_case['system_prompt'][:50]}...")
        print(f"用户输入: {test_case['user_input']}")
        print("\n正在调用API...")
        
        try:
            # 调用API
            result = assistant_node.generate_response(
                api_base_url=api_config["api_base_url"],
                api_key=api_config["api_key"],
                model=api_config["model"],
                system_prompt=test_case["system_prompt"],
                user_input=test_case["user_input"],
                temperature=api_config["temperature"],
                max_tokens=api_config["max_tokens"],
                timeout=api_config["timeout"],
                retry_count=api_config["retry_count"]
            )
            
            generated_text, full_response, status_info = result
            
            print(f"状态: {status_info}")
            
            if generated_text:
                print("\n✅ API调用成功！")
                print("生成的内容:")
                print("-" * 40)
                print(generated_text)
                print("-" * 40)
                
                # 解析完整响应以获取更多信息
                try:
                    response_data = json.loads(full_response)
                    if 'usage' in response_data:
                        usage = response_data['usage']
                        print(f"\nToken使用情况:")
                        print(f"  输入tokens: {usage.get('prompt_tokens', 'N/A')}")
                        print(f"  输出tokens: {usage.get('completion_tokens', 'N/A')}")
                        print(f"  总计tokens: {usage.get('total_tokens', 'N/A')}")
                except:
                    pass
            else:
                print(f"\n❌ API调用失败: {status_info}")
                
        except Exception as e:
            print(f"\n❌ 测试过程中出现异常: {str(e)}")
        
        print("\n" + "="*60)


def test_api_connectivity():
    """测试API连通性"""
    print("\n" + "="*60)
    print("API连通性测试")
    print("="*60)
    
    import requests
    
    api_url = "https://api.ds.lobeapi.com/v1/chat/completions"
    headers = {
        'Content-Type': 'application/json',
        'Authorization': 'Bearer sk-YdPWWj1gQGwgqs95uKtu0AnkWj2OyNp37zhDRf0gyFG2wwVI'
    }
    
    # 简单的测试请求
    payload = {
        "model": "deepseek-r1",
        "messages": [
            {"role": "user", "content": "Hello"}
        ],
        "max_tokens": 50,
        "temperature": 0.7
    }
    
    try:
        print("正在测试API连接...")
        response = requests.post(api_url, headers=headers, json=payload, timeout=10)
        
        print(f"HTTP状态码: {response.status_code}")
        
        if response.status_code == 200:
            print("✅ API连接正常")
            try:
                data = response.json()
                if 'choices' in data:
                    print("✅ 响应格式正确")
                else:
                    print("⚠️ 响应格式异常")
                    print(f"响应内容: {data}")
            except:
                print("⚠️ 响应不是有效的JSON格式")
        else:
            print(f"❌ API连接失败")
            try:
                error_data = response.json()
                print(f"错误信息: {error_data}")
            except:
                print(f"错误响应: {response.text}")
                
    except requests.exceptions.Timeout:
        print("❌ 连接超时")
    except requests.exceptions.ConnectionError:
        print("❌ 连接错误，无法访问API服务")
    except Exception as e:
        print(f"❌ 测试过程中出现异常: {str(e)}")


def main():
    """主测试函数"""
    print("DeepSeek R1 API 配置测试")
    print("测试时间:", __import__('datetime').datetime.now().strftime("%Y-%m-%d %H:%M:%S"))
    
    # 首先测试连通性
    test_api_connectivity()
    
    # 然后测试完整功能
    test_deepseek_api()
    
    print("\n" + "="*60)
    print("🎉 测试完成！")
    print("="*60)
    print("\n使用建议:")
    print("1. 如果测试成功，可以直接在ComfyUI中使用这个配置")
    print("2. 如果遇到问题，请检查网络连接和API密钥")
    print("3. DeepSeek R1在逻辑推理任务上表现出色，可以多尝试这类任务")
    print("4. 建议根据任务类型调整temperature参数")


if __name__ == "__main__":
    main()