"""
Kontext提示词小秘 - 主要节点实现
"""

import json
import requests
import time
from typing import Dict, Any, Tuple, Optional


class KontextPromptAssistant:
    """
    Kontext提示词小秘节点
    支持OpenAI兼容API的提示词处理
    """
    
    def __init__(self):
        pass
    
    @classmethod
    def INPUT_TYPES(cls):
        return {
            "required": {
                "api_base_url": ("STRING", {
                    "default": "https://api.openai.com/v1",
                    "multiline": False,
                    "placeholder": "API基础URL，如: https://api.openai.com/v1"
                }),
                "api_key": ("STRING", {
                    "default": "",
                    "multiline": False,
                    "placeholder": "API密钥",
                    "password": True  # 启用密码模式，自动遮蔽显示
                }),
                "model": ("STRING", {
                    "default": "gpt-3.5-turbo",
                    "multiline": False,
                    "placeholder": "模型名称，如: gpt-3.5-turbo"
                }),
                "system_prompt": ("STRING", {
                    "default": "你是一个有用的AI助手。",
                    "multiline": True,
                    "placeholder": "系统提示词"
                }),
                "user_input": ("STRING", {
                    "default": "",
                    "multiline": True,
                    "placeholder": "用户输入内容"
                }),
                "temperature": ("FLOAT", {
                    "default": 0.7,
                    "min": 0.0,
                    "max": 2.0,
                    "step": 0.1,
                    "display": "slider"
                }),
                "max_tokens": ("INT", {
                    "default": 1000,
                    "min": 1,
                    "max": 4000,
                    "step": 1
                }),
            },
            "optional": {
                "timeout": ("INT", {
                    "default": 30,
                    "min": 5,
                    "max": 120,
                    "step": 1
                }),
                "retry_count": ("INT", {
                    "default": 3,
                    "min": 1,
                    "max": 10,
                    "step": 1
                }),
            }
        }
    
    RETURN_TYPES = ("STRING", "STRING", "STRING")
    RETURN_NAMES = ("generated_text", "full_response", "status_info")
    FUNCTION = "generate_response"
    CATEGORY = "Kontext/提示词工具"
    
    def generate_response(
        self,
        api_base_url: str,
        api_key: str,
        model: str,
        system_prompt: str,
        user_input: str,
        temperature: float,
        max_tokens: int,
        timeout: int = 30,
        retry_count: int = 3
    ) -> Tuple[str, str, str]:
        """
        调用OpenAI兼容API生成响应
        
        Returns:
            Tuple[str, str, str]: (生成的文本, 完整响应JSON, 状态信息)
        """
        
        # 参数验证
        if not api_key.strip():
            return "", "", "错误: API密钥不能为空"
        
        if not user_input.strip():
            return "", "", "错误: 用户输入不能为空"
        
        # 构建请求URL
        api_url = api_base_url.rstrip('/') + '/chat/completions'
        
        # 构建请求头
        headers = {
            'Content-Type': 'application/json',
            'Authorization': f'Bearer {api_key}'
        }
        
        # 构建请求体
        messages = [
            {"role": "system", "content": system_prompt},
            {"role": "user", "content": user_input}
        ]
        
        payload = {
            "model": model,
            "messages": messages,
            "temperature": temperature,
            "max_tokens": max_tokens,
            "stream": False
        }
        
        # 执行API调用（带重试机制）
        for attempt in range(retry_count):
            try:
                print(f"[Kontext提示词小秘] 尝试第 {attempt + 1} 次API调用...")
                
                response = requests.post(
                    api_url,
                    headers=headers,
                    json=payload,
                    timeout=timeout
                )
                
                # 检查HTTP状态码
                if response.status_code == 200:
                    response_data = response.json()
                    
                    # 提取生成的文本
                    if 'choices' in response_data and len(response_data['choices']) > 0:
                        generated_text = response_data['choices'][0]['message']['content']
                        full_response = json.dumps(response_data, ensure_ascii=False, indent=2)
                        status_info = f"成功: 生成了 {len(generated_text)} 个字符"
                        
                        print(f"[Kontext提示词小秘] API调用成功")
                        return generated_text, full_response, status_info
                    else:
                        error_msg = "错误: API响应格式异常，未找到生成的内容"
                        return "", json.dumps(response_data, ensure_ascii=False, indent=2), error_msg
                
                else:
                    # HTTP错误
                    try:
                        error_data = response.json()
                        error_msg = f"API错误 {response.status_code}: {error_data.get('error', {}).get('message', '未知错误')}"
                    except:
                        error_msg = f"HTTP错误 {response.status_code}: {response.text}"
                    
                    if attempt < retry_count - 1:
                        print(f"[Kontext提示词小秘] {error_msg}, 等待重试...")
                        time.sleep(2 ** attempt)  # 指数退避
                        continue
                    else:
                        return "", "", error_msg
            
            except requests.exceptions.Timeout:
                error_msg = f"请求超时 (超过 {timeout} 秒)"
                if attempt < retry_count - 1:
                    print(f"[Kontext提示词小秘] {error_msg}, 等待重试...")
                    time.sleep(2 ** attempt)
                    continue
                else:
                    return "", "", error_msg
            
            except requests.exceptions.ConnectionError:
                error_msg = "连接错误: 无法连接到API服务器"
                if attempt < retry_count - 1:
                    print(f"[Kontext提示词小秘] {error_msg}, 等待重试...")
                    time.sleep(2 ** attempt)
                    continue
                else:
                    return "", "", error_msg
            
            except Exception as e:
                error_msg = f"未知错误: {str(e)}"
                if attempt < retry_count - 1:
                    print(f"[Kontext提示词小秘] {error_msg}, 等待重试...")
                    time.sleep(2 ** attempt)
                    continue
                else:
                    return "", "", error_msg
        
        return "", "", "所有重试都失败了"


class KontextPromptTemplate:
    """
    Kontext提示词模板节点
    提供常用的提示词模板
    """
    
    def __init__(self):
        pass
    
    @classmethod
    def INPUT_TYPES(cls):
        return {
            "required": {
                "template_type": (["创意写作", "代码生成", "文本总结", "翻译助手", "问答助手", "自定义"], {
                    "default": "创意写作"
                }),
                "custom_template": ("STRING", {
                    "default": "",
                    "multiline": True,
                    "placeholder": "自定义模板内容（当选择自定义时使用）"
                }),
                "variables": ("STRING", {
                    "default": "{}",
                    "multiline": True,
                    "placeholder": "JSON格式的变量，如: {\"topic\": \"人工智能\", \"style\": \"正式\"}"
                }),
            }
        }
    
    RETURN_TYPES = ("STRING",)
    RETURN_NAMES = ("prompt_template",)
    FUNCTION = "generate_template"
    CATEGORY = "Kontext/提示词工具"
    
    def generate_template(self, template_type: str, custom_template: str, variables: str) -> Tuple[str]:
        """
        生成提示词模板
        """
        
        # 预定义模板
        templates = {
            "创意写作": """你是一位专业的创意写作助手。请根据以下要求创作内容：

主题：{topic}
风格：{style}
长度：{length}

请确保内容原创、有趣且符合要求。""",
            
            "代码生成": """你是一位专业的程序员。请根据以下需求生成代码：

编程语言：{language}
功能需求：{requirement}
代码风格：{style}

请提供完整、可运行的代码，并包含必要的注释。""",
            
            "文本总结": """你是一位专业的文本分析师。请对以下内容进行总结：

总结类型：{summary_type}
总结长度：{length}
重点关注：{focus}

请提供准确、简洁的总结。""",
            
            "翻译助手": """你是一位专业的翻译专家。请进行以下翻译工作：

源语言：{source_lang}
目标语言：{target_lang}
翻译风格：{style}

请提供准确、自然的翻译结果。""",
            
            "问答助手": """你是一位知识渊博的问答助手。请回答以下问题：

问题领域：{domain}
回答深度：{depth}
回答风格：{style}

请提供准确、有用的回答。"""
        }
        
        # 解析变量
        try:
            vars_dict = json.loads(variables) if variables.strip() else {}
        except json.JSONDecodeError:
            vars_dict = {}
        
        # 选择模板
        if template_type == "自定义":
            template = custom_template
        else:
            template = templates.get(template_type, "")
        
        # 替换变量
        try:
            formatted_template = template.format(**vars_dict)
        except KeyError as e:
            formatted_template = f"模板变量错误: 缺少变量 {e}\n\n原始模板:\n{template}"
        
        return (formatted_template,)


# 导入安全版本节点
try:
    from .secure_nodes import SECURE_NODE_CLASS_MAPPINGS, SECURE_NODE_DISPLAY_NAME_MAPPINGS
    SECURE_NODES_AVAILABLE = True
except ImportError:
    SECURE_NODES_AVAILABLE = False

# 节点映射
NODE_CLASS_MAPPINGS = {
    "KontextPromptAssistant": KontextPromptAssistant,
    "KontextPromptTemplate": KontextPromptTemplate,
}

NODE_DISPLAY_NAME_MAPPINGS = {
    "KontextPromptAssistant": "Kontext提示词小秘",
    "KontextPromptTemplate": "Kontext提示词模板",
}

# 添加安全版本节点（如果可用）
if SECURE_NODES_AVAILABLE:
    NODE_CLASS_MAPPINGS.update(SECURE_NODE_CLASS_MAPPINGS)
    NODE_DISPLAY_NAME_MAPPINGS.update(SECURE_NODE_DISPLAY_NAME_MAPPINGS)