#!/usr/bin/env python3
"""
Kontext提示词小秘 - 安装脚本
自动安装依赖和验证环境
"""

import os
import sys
import subprocess
import importlib.util


def check_python_version():
    """检查Python版本"""
    print("检查Python版本...")
    version = sys.version_info
    if version.major < 3 or (version.major == 3 and version.minor < 7):
        print(f"❌ Python版本过低: {version.major}.{version.minor}")
        print("需要Python 3.7或更高版本")
        return False
    print(f"✅ Python版本: {version.major}.{version.minor}.{version.micro}")
    return True


def install_requirements():
    """安装依赖包"""
    print("\n检查和安装依赖包...")
    
    requirements_file = os.path.join(os.path.dirname(__file__), 'requirements.txt')
    if not os.path.exists(requirements_file):
        print("❌ 未找到requirements.txt文件")
        return False
    
    try:
        # 读取requirements.txt
        with open(requirements_file, 'r', encoding='utf-8') as f:
            requirements = [line.strip() for line in f if line.strip() and not line.startswith('#')]
        
        print(f"需要安装的包: {', '.join(requirements)}")
        
        # 检查是否已安装
        missing_packages = []
        for req in requirements:
            package_name = req.split('>=')[0].split('==')[0].split('<')[0]
            if not check_package_installed(package_name):
                missing_packages.append(req)
        
        if not missing_packages:
            print("✅ 所有依赖包已安装")
            return True
        
        print(f"需要安装: {', '.join(missing_packages)}")
        
        # 安装缺失的包
        for package in missing_packages:
            print(f"正在安装 {package}...")
            result = subprocess.run([
                sys.executable, '-m', 'pip', 'install', package
            ], capture_output=True, text=True)
            
            if result.returncode == 0:
                print(f"✅ {package} 安装成功")
            else:
                print(f"❌ {package} 安装失败:")
                print(result.stderr)
                return False
        
        return True
        
    except Exception as e:
        print(f"❌ 安装依赖时出错: {e}")
        return False


def check_package_installed(package_name):
    """检查包是否已安装"""
    try:
        importlib.import_module(package_name)
        return True
    except ImportError:
        return False


def test_nodes():
    """测试节点功能"""
    print("\n测试节点功能...")
    
    try:
        from nodes import KontextPromptAssistant, KontextPromptTemplate
        
        # 测试模板节点
        template_node = KontextPromptTemplate()
        result = template_node.generate_template(
            template_type="创意写作",
            custom_template="",
            variables='{"topic": "测试", "style": "简洁", "length": "100字"}'
        )
        
        if result and result[0]:
            print("✅ 提示词模板节点测试通过")
        else:
            print("❌ 提示词模板节点测试失败")
            return False
        
        # 测试助手节点（参数验证）
        assistant_node = KontextPromptAssistant()
        result = assistant_node.generate_response(
            api_base_url="https://api.openai.com/v1",
            api_key="",  # 空密钥用于测试验证
            model="gpt-3.5-turbo",
            system_prompt="测试",
            user_input="测试",
            temperature=0.7,
            max_tokens=100
        )
        
        if result[2] == "错误: API密钥不能为空":
            print("✅ 提示词助手节点参数验证通过")
        else:
            print("❌ 提示词助手节点参数验证失败")
            return False
        
        return True
        
    except Exception as e:
        print(f"❌ 节点测试失败: {e}")
        import traceback
        traceback.print_exc()
        return False


def check_comfyui_environment():
    """检查ComfyUI环境"""
    print("\n检查ComfyUI环境...")
    
    # 检查是否在ComfyUI的custom_nodes目录中
    current_dir = os.path.dirname(os.path.abspath(__file__))
    parent_dir = os.path.dirname(current_dir)
    
    if os.path.basename(parent_dir) == "custom_nodes":
        print("✅ 检测到ComfyUI custom_nodes目录")
        comfyui_dir = os.path.dirname(parent_dir)
        print(f"ComfyUI目录: {comfyui_dir}")
        return True
    else:
        print("⚠️  未检测到ComfyUI环境")
        print("请确保将插件放置在ComfyUI/custom_nodes/目录下")
        return False


def create_example_config():
    """创建示例配置文件"""
    print("\n创建示例配置...")
    
    config_content = """# Kontext提示词小秘 - 配置示例
# 将此文件重命名为 user_config.py 并填入你的API信息

# OpenAI API配置
OPENAI_API_KEY = "sk-your-openai-api-key-here"
OPENAI_BASE_URL = "https://api.openai.com/v1"

# Azure OpenAI配置
AZURE_API_KEY = "your-azure-api-key"
AZURE_BASE_URL = "https://your-resource.openai.azure.com/openai/deployments/your-deployment"

# 本地Ollama配置
OLLAMA_BASE_URL = "http://localhost:11434/v1"

# 默认设置
DEFAULT_MODEL = "gpt-3.5-turbo"
DEFAULT_TEMPERATURE = 0.7
DEFAULT_MAX_TOKENS = 1000
"""
    
    config_file = os.path.join(os.path.dirname(__file__), 'config_example.py')
    try:
        with open(config_file, 'w', encoding='utf-8') as f:
            f.write(config_content)
        print(f"✅ 示例配置文件已创建: {config_file}")
        return True
    except Exception as e:
        print(f"❌ 创建配置文件失败: {e}")
        return False


def main():
    """主安装函数"""
    print("=" * 60)
    print("Kontext提示词小秘 - 安装向导")
    print("=" * 60)
    
    success = True
    
    # 检查Python版本
    if not check_python_version():
        success = False
    
    # 检查ComfyUI环境
    check_comfyui_environment()
    
    # 安装依赖
    if not install_requirements():
        success = False
    
    # 测试节点
    if not test_nodes():
        success = False
    
    # 创建示例配置
    create_example_config()
    
    print("\n" + "=" * 60)
    if success:
        print("🎉 安装完成！")
        print("=" * 60)
        print("\n下一步:")
        print("1. 重启ComfyUI")
        print("2. 在节点菜单中找到 'Kontext/提示词工具' 分类")
        print("3. 配置你的API密钥和模型")
        print("4. 开始使用Kontext提示词小秘！")
        print("\n📚 查看README.md获取详细使用说明")
        print("🔧 查看examples/目录获取配置示例")
    else:
        print("❌ 安装过程中遇到问题")
        print("=" * 60)
        print("\n请检查上述错误信息并重试")
        print("如需帮助，请查看README.md或提交Issue")
    
    print("\n感谢使用Kontext提示词小秘！")


if __name__ == "__main__":
    main()