# Kontext提示词小秘 - 安全使用指南

## 🔒 API密钥安全问题

### ⚠️ 安全风险
1. **界面显示**: API密钥在ComfyUI界面中明文显示
2. **工作流导出**: 导出的JSON文件包含完整API密钥
3. **分享风险**: 分享工作流时可能泄露密钥
4. **日志记录**: 可能在日志中记录敏感信息

## 🛡️ 安全解决方案

### 方案1: 使用安全版本节点（推荐）

#### 新增节点功能:
- **Kontext提示词小秘(安全版)**: 带密钥保护的主节点
- **Kontext密钥管理器**: 专门的密钥管理工具

#### 安全特性:
```
✅ 密码模式输入 (自动星号遮蔽)
✅ 环境变量支持
✅ 敏感信息过滤
✅ 密钥验证功能
✅ 安全的响应遮蔽
```

### 方案2: 环境变量配置

#### 设置步骤:
1. **创建环境变量**:
   ```bash
   # Windows
   set OPENAI_API_KEY=your-api-key-here
   
   # Linux/Mac
   export OPENAI_API_KEY=your-api-key-here
   ```

2. **在节点中使用**:
   ```
   ✅ 勾选 "使用环境变量"
   ✅ 环境变量名: OPENAI_API_KEY
   ✅ API密钥输入框: 留空
   ```

### 方案3: 配置文件管理

#### 创建安全配置文件:
```python
# user_config.py (不要提交到版本控制)
API_KEYS = {
    "openai": "sk-your-openai-key",
    "deepseek": "sk-your-deepseek-key",
    "claude": "sk-ant-your-claude-key"
}
```

## 🔧 实际操作指南

### 使用安全版本节点

#### 步骤1: 添加密钥管理器
```
1. 添加节点: Kontext密钥管理器
2. 输入API密钥 (会自动遮蔽显示)
3. 选择操作: "验证密钥"
4. 查看验证结果
```

#### 步骤2: 使用安全版主节点
```
1. 添加节点: Kontext提示词小秘(安全版)
2. 配置API信息
3. 勾选 "使用环境变量" (推荐)
4. 或在密钥输入框输入 (会自动遮蔽)
```

### 环境变量配置方法

#### Windows系统:
```cmd
# 临时设置 (当前会话有效)
set OPENAI_API_KEY=sk-your-key-here
set DEEPSEEK_API_KEY=sk-your-deepseek-key

# 永久设置
setx OPENAI_API_KEY "sk-your-key-here"
```

#### Linux/Mac系统:
```bash
# 临时设置
export OPENAI_API_KEY=sk-your-key-here

# 永久设置 (添加到 ~/.bashrc 或 ~/.zshrc)
echo 'export OPENAI_API_KEY=sk-your-key-here' >> ~/.bashrc
source ~/.bashrc
```

#### ComfyUI启动脚本:
```bash
#!/bin/bash
# start_comfyui_secure.sh

# 设置API密钥
export OPENAI_API_KEY="sk-your-openai-key"
export DEEPSEEK_API_KEY="sk-your-deepseek-key"
export CLAUDE_API_KEY="sk-ant-your-claude-key"

# 启动ComfyUI
python main.py --listen
```

## 🚨 工作流分享安全

### 导出前检查清单:
```
□ 检查所有API密钥输入框是否为空
□ 确认使用环境变量模式
□ 验证导出的JSON不包含敏感信息
□ 测试工作流在新环境中的运行
```

### 安全的工作流分享方式:

#### 方法1: 清理版本
```json
{
  "nodes": [
    {
      "type": "KontextPromptAssistant",
      "widgets_values": [
        "https://api.openai.com/v1",  // API URL 可以保留
        "",                          // API密钥 - 清空
        "gpt-3.5-turbo",            // 模型名称 可以保留
        "系统提示词...",              // 提示词 可以保留
        "用户输入...",               // 用户输入 可以保留
        0.7,                        // 其他参数 可以保留
        1000
      ]
    }
  ]
}
```

#### 方法2: 模板化
```json
{
  "api_key": "请设置你的API密钥",
  "instructions": "使用前请配置API密钥",
  "setup_guide": "参见SECURITY_GUIDE.md"
}
```

## 🔍 安全检查工具

### 自动检查脚本:
```python
# security_check.py
import json
import re

def check_workflow_security(workflow_file):
    """检查工作流文件的安全性"""
    with open(workflow_file, 'r') as f:
        data = json.load(f)
    
    issues = []
    
    # 检查API密钥
    api_key_pattern = r'sk-[a-zA-Z0-9]{20,}'
    if re.search(api_key_pattern, str(data)):
        issues.append("⚠️ 发现可能的API密钥")
    
    # 检查敏感信息
    sensitive_patterns = [
        r'Bearer [a-zA-Z0-9]{20,}',
        r'password.*[a-zA-Z0-9]{8,}',
        r'token.*[a-zA-Z0-9]{20,}'
    ]
    
    for pattern in sensitive_patterns:
        if re.search(pattern, str(data), re.IGNORECASE):
            issues.append(f"⚠️ 发现敏感信息模式: {pattern}")
    
    return issues

# 使用示例
issues = check_workflow_security("my_workflow.json")
if issues:
    print("安全检查发现问题:")
    for issue in issues:
        print(f"  {issue}")
else:
    print("✅ 工作流安全检查通过")
```

## 💡 最佳实践

### 1. 开发环境
```
✅ 使用环境变量存储API密钥
✅ 在.gitignore中排除配置文件
✅ 定期轮换API密钥
✅ 使用最小权限原则
```

### 2. 生产环境
```
✅ 使用专用的API密钥
✅ 设置使用量监控和警报
✅ 定期审计API使用情况
✅ 实施访问控制
```

### 3. 分享工作流
```
✅ 使用安全版本节点
✅ 清理所有敏感信息
✅ 提供配置说明文档
✅ 包含安全使用指南
```

## 🛠️ 故障排除

### 常见问题:

#### 1. 环境变量未生效
```
问题: 设置了环境变量但节点无法读取
解决: 
- 重启ComfyUI
- 检查变量名拼写
- 确认变量作用域
```

#### 2. 密钥验证失败
```
问题: API密钥验证不通过
解决:
- 检查密钥是否正确
- 确认API服务可用
- 验证网络连接
```

#### 3. 工作流导入问题
```
问题: 导入工作流后无法运行
解决:
- 重新配置API密钥
- 检查节点版本兼容性
- 查看错误日志
```

## 📋 安全检查清单

### 使用前检查:
- [ ] API密钥已安全配置
- [ ] 使用环境变量或安全版本节点
- [ ] 网络连接安全
- [ ] 了解API使用成本

### 分享前检查:
- [ ] 清除所有API密钥
- [ ] 检查JSON文件内容
- [ ] 提供配置说明
- [ ] 测试工作流可用性

### 定期维护:
- [ ] 轮换API密钥
- [ ] 检查使用量
- [ ] 更新安全配置
- [ ] 审查访问日志

---

**记住: 安全是一个持续的过程，不是一次性的设置！** 🔒