<?php
session_start();

// 检查用户是否已登录
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php?error=请先登录");
    exit;
}

// 检查是否提供了举报ID
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: user_dashboard.php?error=无效的举报ID");
    exit;
}

require_once 'db_connect.php';

// 确保数据库连接变量名正确
if (!isset($pdo) && isset($conn)) {
    $pdo = $conn;
}

$report_id = (int)$_GET['id'];
$user_id = $_SESSION['user_id'];

// 获取举报详情
$report = null;
try {
    // 根据数据库连接类型选择不同的查询方式
    if ($pdo instanceof PDO) {
        // PDO方式
        $stmt = $pdo->prepare("SELECT * FROM reports WHERE id = ? AND user_id = ?");
        $stmt->execute([$report_id, $user_id]);
        $report = $stmt->fetch(PDO::FETCH_ASSOC);
    } else {
        // mysqli方式
        $stmt = $pdo->prepare("SELECT * FROM reports WHERE id = ? AND user_id = ?");
        $stmt->bind_param("ii", $report_id, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $report = $result->fetch_assoc();
    }
    
    // 如果找不到举报或不属于当前用户
    if (!$report) {
        header("Location: user_dashboard.php?error=找不到该举报或您无权查看");
        exit;
    }
} catch (Exception $e) {
    header("Location: user_dashboard.php?error=获取举报详情失败: " . $e->getMessage());
    exit;
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>举报详情 - 举报系统</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            padding-top: 20px;
            padding-bottom: 20px;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            background-attachment: fixed;
            position: relative;
        }
        
        body::before {
            content: "";
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-image: url('https://cdn.pixabay.com/photo/2018/03/22/02/37/background-3249063_1280.png');
            background-size: cover;
            background-position: center;
            opacity: 0.1;
            z-index: -1;
        }
        
        .container {
            max-width: 800px;
            position: relative;
            z-index: 1;
        }
        
        .card {
            border-radius: 10px;
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
            border: none;
            background-color: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(5px);
        }
        
        .card-body {
            padding: 30px;
        }
        
        h1 {
            color: #0d6efd;
            font-weight: 600;
            text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.1);
        }
        
        .btn-back {
            color: #6c757d;
            text-decoration: none;
            transition: color 0.3s;
            display: inline-flex;
            align-items: center;
        }
        
        .btn-back:hover {
            color: #0d6efd;
        }
        
        .report-info-item {
            margin-bottom: 20px;
        }
        
        .report-info-item .label {
            font-weight: 600;
            color: #495057;
            margin-bottom: 5px;
        }
        
        .report-info-item .value {
            color: #212529;
            background-color: rgba(248, 249, 250, 0.7);
            padding: 10px 15px;
            border-radius: 8px;
            border: 1px solid #dee2e6;
        }
        
        .badge {
            padding: 8px 12px;
            font-weight: 500;
            border-radius: 6px;
            font-size: 0.9rem;
        }
        
        .badge.bg-pending { background-color: #6c757d; }
        .badge.bg-processing { background-color: #0d6efd; }
        .badge.bg-resolved { background-color: #198754; }
        .badge.bg-rejected { background-color: #dc3545; }
        
        .evidence-image {
            max-width: 100%;
            max-height: 300px;
            border-radius: 8px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        .admin-comment {
            background-color: rgba(13, 110, 253, 0.1);
            border-left: 4px solid #0d6efd;
            padding: 15px;
            border-radius: 0 8px 8px 0;
            margin-top: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1>举报详情</h1>
            <a href="user_dashboard.php" class="btn-back">
                <i class="bi bi-arrow-left me-2"></i> 返回用户平台
            </a>
        </div>
        
        <div class="card mb-4">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h5 class="card-title mb-0">举报 #<?php echo $report['id']; ?></h5>
                    <?php
                    $status_text = '';
                    $status_class = '';
                    switch ($report['status']) {
                        case 'pending':
                            $status_text = '待处理';
                            $status_class = 'bg-pending';
                            break;
                        case 'processing':
                            $status_text = '处理中';
                            $status_class = 'bg-processing';
                            break;
                        case 'resolved':
                            $status_text = '已解决';
                            $status_class = 'bg-resolved';
                            break;
                        case 'rejected':
                            $status_text = '已驳回';
                            $status_class = 'bg-rejected';
                            break;
                    }
                    ?>
                    <span class="badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                </div>
                
                <div class="report-info-item">
                    <div class="label">举报类型</div>
                    <div class="value"><?php echo htmlspecialchars($report['report_type']); ?></div>
                </div>
                
                <div class="report-info-item">
                    <div class="label">举报人</div>
                    <div class="value"><?php echo htmlspecialchars($report['reporter_name']); ?></div>
                </div>
                
                <div class="report-info-item">
                    <div class="label">联系方式</div>
                    <div class="value"><?php echo htmlspecialchars($report['reporter_contact']); ?></div>
                </div>
                
                <div class="report-info-item">
                    <div class="label">举报内容</div>
                    <div class="value"><?php echo nl2br(htmlspecialchars($report['report_content'])); ?></div>
                </div>
                
                <?php if (!empty($report['reported_url'])): ?>
                <div class="report-info-item">
                    <div class="label">相关链接</div>
                    <div class="value">
                        <a href="<?php echo htmlspecialchars($report['reported_url']); ?>" target="_blank">
                            <?php echo htmlspecialchars($report['reported_url']); ?>
                            <i class="bi bi-box-arrow-up-right ms-1"></i>
                        </a>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($report['evidence_file'])): ?>
                <div class="report-info-item">
                    <div class="label">证据文件</div>
                    <div class="value">
                        <?php
                        $file_extension = strtolower(pathinfo($report['evidence_file'], PATHINFO_EXTENSION));
                        $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp'];
                        
                        if (in_array($file_extension, $image_extensions)):
                        ?>
                            <img src="<?php echo htmlspecialchars($report['evidence_file']); ?>" alt="证据图片" class="evidence-image">
                        <?php else: ?>
                            <a href="<?php echo htmlspecialchars($report['evidence_file']); ?>" class="btn btn-outline-primary" download>
                                <i class="bi bi-download me-2"></i>下载证据文件
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <div class="report-info-item">
                    <div class="label">提交时间</div>
                    <div class="value"><?php echo date('Y-m-d H:i:s', strtotime($report['created_at'])); ?></div>
                </div>
                
                <?php if (!empty($report['admin_comment'])): ?>
                <div class="report-info-item">
                    <div class="label">管理员回复</div>
                    <div class="value">
                        <div class="admin-comment">
                            <?php echo nl2br(htmlspecialchars($report['admin_comment'])); ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($report['updated_at']) && $report['updated_at'] != $report['created_at']): ?>
                <div class="report-info-item">
                    <div class="label">最后更新时间</div>
                    <div class="value"><?php echo date('Y-m-d H:i:s', strtotime($report['updated_at'])); ?></div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <footer class="mt-5 text-center text-muted">
            <small>&copy; <?php echo date('Y'); ?> 举报系统 - 共建健康网络环境</small>
        </footer>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>