<?php
session_start();

// 检查用户是否已登录
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php?error=请先登录");
    exit;
}

require_once 'db_connect.php';

// 获取用户信息
$user_id = $_SESSION['user_id'];
$username = $_SESSION['username'];

// 确保数据库连接变量名正确
if (!isset($pdo) && isset($conn)) {
    $pdo = $conn;
}

// 获取用户提交的举报列表
$reports = [];
try {
    // 根据数据库连接类型选择不同的查询方式
    if ($pdo instanceof PDO) {
        // PDO方式
        $stmt = $pdo->prepare("SELECT * FROM reports WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$user_id]);
        $reports = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } else {
        // mysqli方式
        $stmt = $pdo->prepare("SELECT * FROM reports WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $reports = [];
        while ($row = $result->fetch_assoc()) {
            $reports[] = $row;
        }
    }
} catch (Exception $e) {
    $error_message = "获取举报列表失败: " . $e->getMessage();
}

// 获取用户详细信息
$user_info = [];
try {
    if ($pdo instanceof PDO) {
        // PDO方式
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user_info = $stmt->fetch(PDO::FETCH_ASSOC);
    } else {
        // mysqli方式
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user_info = $result->fetch_assoc();
    }
} catch (Exception $e) {
    $error_message = "获取用户信息失败: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>用户平台 - 举报系统</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            padding-top: 20px;
            padding-bottom: 20px;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            background-attachment: fixed;
            position: relative;
        }
        
        body::before {
            content: "";
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-image: url('https://cdn.pixabay.com/photo/2018/03/22/02/37/background-3249063_1280.png');
            background-size: cover;
            background-position: center;
            opacity: 0.1;
            z-index: -1;
        }
        
        .container {
            max-width: 900px;
            position: relative;
            z-index: 1;
        }
        
        .card {
            border-radius: 10px;
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
            border: none;
            background-color: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(5px);
            margin-bottom: 20px;
        }
        
        .card-body {
            padding: 25px;
        }
        
        h1, h2, h3 {
            color: #0d6efd;
            font-weight: 600;
            text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.1);
        }
        
        .nav-tabs .nav-link {
            border: none;
            color: #495057;
            font-weight: 500;
            padding: 12px 20px;
            border-radius: 0;
            transition: all 0.3s;
        }
        
        .nav-tabs .nav-link.active {
            color: #0d6efd;
            background-color: transparent;
            border-bottom: 3px solid #0d6efd;
        }
        
        .nav-tabs .nav-link:hover:not(.active) {
            background-color: rgba(13, 110, 253, 0.05);
            border-bottom: 3px solid rgba(13, 110, 253, 0.3);
        }
        
        .badge {
            padding: 6px 10px;
            font-weight: 500;
            border-radius: 6px;
        }
        
        .badge.bg-pending { background-color: #6c757d; }
        .badge.bg-processing { background-color: #0d6efd; }
        .badge.bg-resolved { background-color: #198754; }
        .badge.bg-rejected { background-color: #dc3545; }
        
        .btn-back {
            color: #6c757d;
            text-decoration: none;
            transition: color 0.3s;
            display: inline-flex;
            align-items: center;
        }
        
        .btn-back:hover {
            color: #0d6efd;
        }
        
        .user-info-item {
            margin-bottom: 15px;
        }
        
        .user-info-item .label {
            font-weight: 600;
            color: #495057;
        }
        
        .user-info-item .value {
            color: #212529;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
        }
        
        .empty-state i {
            font-size: 3rem;
            color: #6c757d;
            margin-bottom: 15px;
        }
        
        .empty-state p {
            color: #6c757d;
            font-size: 1.1rem;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1>用户平台</h1>
            <a href="index.php" class="btn-back">
                <i class="bi bi-arrow-left me-2"></i> 返回首页
            </a>
        </div>
        
        <?php if(isset($error_message)): ?>
        <div class="alert alert-danger">
            <i class="bi bi-exclamation-triangle-fill me-2"></i>
            <?php echo htmlspecialchars($error_message); ?>
        </div>
        <?php endif; ?>
        
        <ul class="nav nav-tabs mb-4" id="userTabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link active" id="reports-tab" data-bs-toggle="tab" data-bs-target="#reports-tab-pane" type="button" role="tab" aria-controls="reports-tab-pane" aria-selected="true">
                    <i class="bi bi-flag me-2"></i>我的举报
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="profile-tab" data-bs-toggle="tab" data-bs-target="#profile-tab-pane" type="button" role="tab" aria-controls="profile-tab-pane" aria-selected="false">
                    <i class="bi bi-person me-2"></i>个人信息
                </button>
            </li>
        </ul>
        
        <div class="tab-content" id="userTabsContent">
            <!-- 我的举报标签页 -->
            <div class="tab-pane fade show active" id="reports-tab-pane" role="tabpanel" aria-labelledby="reports-tab" tabindex="0">
                <div class="card">
                    <div class="card-body">
                        <h2 class="card-title mb-4">我的举报记录</h2>
                        
                        <?php if(count($reports) > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>类型</th>
                                        <th>内容摘要</th>
                                        <th>提交时间</th>
                                        <th>状态</th>
                                        <th>操作</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($reports as $report): ?>
                                    <tr>
                                        <td><?php echo $report['id']; ?></td>
                                        <td><?php echo htmlspecialchars($report['report_type']); ?></td>
                                        <td><?php echo htmlspecialchars(mb_substr($report['report_content'], 0, 30)) . (mb_strlen($report['report_content']) > 30 ? '...' : ''); ?></td>
                                        <td><?php echo date('Y-m-d H:i', strtotime($report['created_at'])); ?></td>
                                        <td>
                                            <?php
                                            $status_text = '';
                                            $status_class = '';
                                            switch ($report['status']) {
                                                case 'pending':
                                                    $status_text = '待处理';
                                                    $status_class = 'bg-pending';
                                                    break;
                                                case 'processing':
                                                    $status_text = '处理中';
                                                    $status_class = 'bg-processing';
                                                    break;
                                                case 'resolved':
                                                    $status_text = '已解决';
                                                    $status_class = 'bg-resolved';
                                                    break;
                                                case 'rejected':
                                                    $status_text = '已驳回';
                                                    $status_class = 'bg-rejected';
                                                    break;
                                            }
                                            ?>
                                            <span class="badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                                        </td>
                                        <td>
                                            <a href="view_user_report.php?id=<?php echo $report['id']; ?>" class="btn btn-sm btn-primary">查看详情</a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <div class="empty-state">
                            <i class="bi bi-inbox"></i>
                            <p>您还没有提交过举报</p>
                            <a href="index.php" class="btn btn-primary mt-3">去提交举报</a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- 个人信息标签页 -->
            <div class="tab-pane fade" id="profile-tab-pane" role="tabpanel" aria-labelledby="profile-tab" tabindex="0">
                <div class="card">
                    <div class="card-body">
                        <h2 class="card-title mb-4">个人信息</h2>
                        
                        <?php if($user_info): ?>
                        <div class="row">
                            <div class="col-md-8 offset-md-2">
                                <div class="text-center mb-4">
                                    <div class="display-1 text-primary">
                                        <i class="bi bi-person-circle"></i>
                                    </div>
                                    <h3><?php echo htmlspecialchars($user_info['username']); ?></h3>
                                </div>
                                
                                <div class="user-info-item">
                                    <div class="label">用户ID</div>
                                    <div class="value"><?php echo $user_info['id']; ?></div>
                                </div>
                                
                                <div class="user-info-item">
                                    <div class="label">电子邮箱</div>
                                    <div class="value"><?php echo htmlspecialchars($user_info['email']); ?></div>
                                </div>
                                
                                <div class="user-info-item">
                                    <div class="label">注册时间</div>
                                    <div class="value"><?php echo date('Y-m-d H:i:s', strtotime($user_info['created_at'])); ?></div>
                                </div>
                                
                                <div class="user-info-item">
                                    <div class="label">举报数量</div>
                                    <div class="value"><?php echo count($reports); ?></div>
                                </div>
                                
                                <div class="mt-4 text-center">
                                    <a href="edit_profile.php" class="btn btn-primary">
                                        <i class="bi bi-pencil me-2"></i>修改个人信息
                                    </a>
                                    <a href="change_password.php" class="btn btn-outline-secondary ms-2">
                                        <i class="bi bi-key me-2"></i>修改密码
                                    </a>
                                </div>
                            </div>
                        </div>
                        <?php else: ?>
                        <div class="alert alert-danger">
                            <i class="bi bi-exclamation-triangle-fill me-2"></i>
                            无法获取用户信息
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        
        <footer class="mt-5 text-center text-muted">
            <small>&copy; <?php echo date('Y'); ?> 举报系统 - 共建健康网络环境</small>
        </footer>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>