<?php
session_start();
require_once 'db_connect.php';

// 确保数据库连接变量名正确
// 如果db_connect.php中使用的是$conn，我们需要将其赋值给$pdo
if (!isset($pdo) && isset($conn)) {
    $pdo = $conn;
}

// 如果仍然没有有效的数据库连接，显示错误
if (!isset($pdo)) {
    die("数据库连接失败：db_connect.php 中没有定义 \$pdo 变量");
}

// 检查是否提交了表单
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    
    // 验证输入
    if (empty($username) || empty($email) || empty($password) || empty($confirm_password)) {
        header("Location: register.php?error=请填写所有必填字段");
        exit;
    }
    
    // 验证用户名格式
    if (!preg_match('/^[a-zA-Z0-9_]{3,20}$/', $username)) {
        header("Location: register.php?error=用户名格式不正确，只能包含字母、数字和下划线，长度为3-20个字符");
        exit;
    }
    
    // 验证邮箱格式
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        header("Location: register.php?error=邮箱格式不正确");
        exit;
    }
    
    // 验证密码长度
    if (strlen($password) < 6) {
        header("Location: register.php?error=密码长度至少为6个字符");
        exit;
    }
    
    // 验证两次密码是否一致
    if ($password !== $confirm_password) {
        header("Location: register.php?error=两次输入的密码不一致");
        exit;
    }
    
    try {
        // 检查用户名是否已存在 - 修改为mysqli风格
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM users WHERE username = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        if ($row['count'] > 0) {
            header("Location: register.php?error=用户名已被使用");
            exit;
        }
        
        // 检查邮箱是否已存在 - 修改为mysqli风格
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        if ($row['count'] > 0) {
            header("Location: register.php?error=邮箱已被注册");
            exit;
        }
        
        // 密码加密
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        // 插入新用户 - 修改为mysqli风格
        $stmt = $pdo->prepare("INSERT INTO users (username, email, password, created_at) VALUES (?, ?, ?, NOW())");
        $stmt->bind_param("sss", $username, $email, $hashed_password);
        $stmt->execute();
        
        // 获取新用户ID - 修改为mysqli风格
        $user_id = $pdo->insert_id;
        
        // 设置会话，自动登录
        $_SESSION['user_id'] = $user_id;
        $_SESSION['username'] = $username;
        
        // 重定向到首页
        header("Location: index.php");
        exit;
    } catch (Exception $e) {
        // 修改为捕获所有异常，而不仅仅是PDOException
        header("Location: register.php?error=系统错误，请稍后再试");
        exit;
    }
} else {
    // 如果不是POST请求，重定向到注册页面
    header("Location: register.php");
    exit;
}
?>