<?php
/**
 * 通知辅助函数
 * 用于发送各种类型的通知
 */

// 引入PHPMailer库
require_once __DIR__ . '/../vendor/autoload.php';
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use PHPMailer\PHPMailer\SMTP;

/**
 * 发送邮件通知
 * 
 * @param string $to 收件人邮箱
 * @param string $subject 邮件主题
 * @param string $message 邮件内容
 * @return bool 是否发送成功
 */
function send_email_notification($to, $subject, $message) {
    if (empty($to)) {
        error_log('邮件发送失败: 收件人邮箱为空');
        return false;
    }
    
    // 加载通知设置
    $settings = [];
    $config_file = __DIR__ . '/../config/notification_settings.php';
    if (file_exists($config_file)) {
        try {
            $settings = require $config_file;
        } catch (Exception $e) {
            error_log('加载通知设置失败: ' . $e->getMessage());
            return false;
        }
    }
    
    // 检查是否使用SMTP
    if (isset($settings['use_smtp']) && $settings['use_smtp'] && 
        isset($settings['smtp_host']) && !empty($settings['smtp_host']) &&
        isset($settings['smtp_username']) && !empty($settings['smtp_username']) &&
        isset($settings['smtp_password']) && !empty($settings['smtp_password'])) {
        
        // 使用PHPMailer通过SMTP发送
        return send_email_via_smtp($to, $subject, $message, $settings);
    } else {
        // 使用PHP内置mail()函数发送
        return send_email_via_mail($to, $subject, $message);
    }
}

/**
 * 使用PHP内置mail()函数发送邮件
 * 
 * @param string $to 收件人邮箱
 * @param string $subject 邮件主题
 * @param string $message 邮件内容
 * @return bool 是否发送成功
 */
function send_email_via_mail($to, $subject, $message) {
    // 设置邮件头信息
    $headers = [
        'MIME-Version: 1.0',
        'Content-type: text/html; charset=UTF-8',
        'From: 举报系统 <noreply@' . $_SERVER['HTTP_HOST'] . '>',
        'Reply-To: noreply@' . $_SERVER['HTTP_HOST'],
        'X-Mailer: PHP/' . phpversion()
    ];
    
    // 尝试发送邮件
    $result = mail($to, $subject, $message, implode("\r\n", $headers));
    
    if (!$result) {
        $error = error_get_last();
        error_log('邮件发送失败: ' . ($error ? $error['message'] : '未知错误'));
    }
    
    return $result;
}

/**
 * 使用PHPMailer通过SMTP发送邮件
 * 
 * @param string $to 收件人邮箱
 * @param string $subject 邮件主题
 * @param string $message 邮件内容
 * @param array $settings SMTP设置
 * @return bool 是否发送成功
 */
function send_email_via_smtp($to, $subject, $message, $settings) {
    $mail = new PHPMailer(true);
    
    try {
        // 服务器设置
        $mail->isSMTP();
        $mail->Host = $settings['smtp_host'];
        $mail->SMTPAuth = true;
        $mail->Username = $settings['smtp_username'];
        $mail->Password = $settings['smtp_password'];
        
        // 设置加密方式
        if (isset($settings['smtp_secure']) && $settings['smtp_secure'] !== 'none') {
            $mail->SMTPSecure = $settings['smtp_secure'];
        }
        
        // 设置端口
        $mail->Port = isset($settings['smtp_port']) ? (int)$settings['smtp_port'] : 465;
        
        // 调试级别，0为关闭，4为最详细
        $mail->SMTPDebug = 0;
        
        // 发件人设置
        $sender_email = isset($settings['smtp_sender']) && !empty($settings['smtp_sender']) 
            ? $settings['smtp_sender'] 
            : $settings['smtp_username'];
            
        $sender_name = isset($settings['smtp_sender_name']) && !empty($settings['smtp_sender_name'])
            ? $settings['smtp_sender_name']
            : '举报系统';
            
        $mail->setFrom($sender_email, $sender_name);
        $mail->addReplyTo($sender_email, $sender_name);
        
        // 收件人设置
        $mail->addAddress($to);
        
        // 内容设置
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $message;
        $mail->CharSet = 'UTF-8';
        
        // 发送邮件
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log('PHPMailer发送邮件失败: ' . $mail->ErrorInfo);
        return false;
    }
}

/**
 * 发送企业微信机器人通知
 * 
 * @param string $webhook_url 企业微信机器人Webhook地址
 * @param string $title 通知标题
 * @param string $content 通知内容
 * @param array $data 附加数据
 * @return bool 是否发送成功
 */
function send_wechat_work_notification($webhook_url, $title, $content, $data = []) {
    if (empty($webhook_url)) {
        return false;
    }
    
    // 构建Markdown格式消息
    $markdown_content = "# {$title}\n\n";
    $markdown_content .= "> {$content}\n\n";
    
    // 添加数据项
    if (!empty($data)) {
        foreach ($data as $key => $value) {
            $markdown_content .= "- **{$key}**：{$value}\n";
        }
    }
    
    // 添加时间戳
    $markdown_content .= "\n*发送时间：" . date('Y-m-d H:i:s') . "*";
    
    // 构建消息体
    $message = [
        'msgtype' => 'markdown',
        'markdown' => [
            'content' => $markdown_content
        ]
    ];
    
    // 发送请求到企业微信API
    $ch = curl_init($webhook_url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        error_log("企业微信通知发送失败: " . $error);
        return false;
    }
    
    $response_data = json_decode($response, true);
    
    // 企业微信API返回0表示成功
    if (isset($response_data['errcode']) && $response_data['errcode'] === 0) {
        return true;
    } else {
        $error_message = isset($response_data['errmsg']) ? $response_data['errmsg'] : '未知错误';
        error_log("企业微信API返回错误: " . $error_message);
        return false;
    }
}

/**
 * 发送新举报通知
 * 
 * @param array $report 举报信息
 * @return void
 */
function send_new_report_notification($report) {
    // 加载通知设置
    $settings = [];
    $config_file = __DIR__ . '/../config/notification_settings.php';
    if (file_exists($config_file)) {
        try {
            $settings = require $config_file;
        } catch (Exception $e) {
            error_log('加载通知设置失败: ' . $e->getMessage());
            return;
        }
    }
    
    // 检查是否启用邮件通知
    if (isset($settings['enable_email']) && $settings['enable_email'] && 
        isset($settings['admin_email']) && !empty($settings['admin_email'])) {
        
        // 构建邮件内容
        $subject = '【举报系统】新举报提交通知';
        
        $message = '
        <html>
        <head>
            <title>新举报提交通知</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background-color: #f8f9fa; padding: 15px; border-bottom: 3px solid #0d6efd; }
                .content { padding: 20px 0; }
                .footer { font-size: 12px; color: #6c757d; border-top: 1px solid #dee2e6; padding-top: 15px; margin-top: 20px; }
                table { width: 100%; border-collapse: collapse; }
                table, th, td { border: 1px solid #dee2e6; }
                th, td { padding: 10px; text-align: left; }
                th { background-color: #f8f9fa; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h2>新举报提交通知</h2>
                </div>
                <div class="content">
                    <p>系统收到一条新的举报信息，请及时处理。</p>
                    
                    <table>
                        <tr>
                            <th>举报ID</th>
                            <td>' . $report['id'] . '</td>
                        </tr>
                        <tr>
                            <th>举报类型</th>
                            <td>' . htmlspecialchars($report['report_type']) . '</td>
                        </tr>
                        <tr>
                            <th>举报人</th>
                            <td>' . htmlspecialchars($report['reporter_name']) . '</td>
                        </tr>
                        <tr>
                            <th>提交时间</th>
                            <td>' . date('Y-m-d H:i:s', strtotime($report['created_at'])) . '</td>
                        </tr>
                        <tr>
                            <th>状态</th>
                            <td>待处理</td>
                        </tr>
                    </table>
                    
                    <p style="margin-top: 20px;">
                        <a href="' . (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]/admin/view_report.php?id=" . $report['id'] . '" 
                           style="background-color: #0d6efd; color: white; padding: 10px 15px; text-decoration: none; border-radius: 4px;">
                            查看详情
                        </a>
                    </p>
                </div>
                <div class="footer">
                    <p>此邮件由系统自动发送，请勿直接回复。</p>
                    <p>发送时间：' . date('Y-m-d H:i:s') . '</p>
                </div>
            </div>
        </body>
        </html>';
        
        // 发送邮件通知
        send_email_notification($settings['admin_email'], $subject, $message);
    }
    
    // 检查是否启用企业微信通知
    if (isset($settings['enable_wechat_work']) && $settings['enable_wechat_work'] && 
        isset($settings['wechat_work_webhook']) && !empty($settings['wechat_work_webhook'])) {
        
        // 构建通知数据
        $data = [
            '举报ID' => $report['id'],
            '举报类型' => $report['report_type'],
            '举报人' => $report['reporter_name'],
            '提交时间' => date('Y-m-d H:i:s', strtotime($report['created_at'])),
            '状态' => '待处理'
        ];
        
        // 发送企业微信通知
        send_wechat_work_notification(
            $settings['wechat_work_webhook'],
            '新举报提交通知',
            '系统收到一条新的举报信息，请及时处理。',
            $data
        );
    }
}