<?php
session_start();

// 检查用户是否已登录
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php?error=请先登录");
    exit;
}

require_once 'db_connect.php';

// 确保数据库连接变量名正确
if (!isset($pdo) && isset($conn)) {
    $pdo = $conn;
}

$user_id = $_SESSION['user_id'];
$success_message = '';
$error_message = '';

// 获取用户信息
try {
    if ($pdo instanceof PDO) {
        // PDO方式
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
    } else {
        // mysqli方式
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
    }
    
    if (!$user) {
        header("Location: user_dashboard.php?error=找不到用户信息");
        exit;
    }
} catch (Exception $e) {
    $error_message = "获取用户信息失败: " . $e->getMessage();
}

// 处理表单提交
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    
    // 验证输入
    if (empty($username) || empty($email)) {
        $error_message = "用户名和邮箱不能为空";
    } elseif (!preg_match('/^[a-zA-Z0-9_]{3,20}$/', $username)) {
        $error_message = "用户名格式不正确，只能包含字母、数字和下划线，长度为3-20个字符";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = "邮箱格式不正确";
    } else {
        try {
            // 检查用户名是否已被其他用户使用
            if ($pdo instanceof PDO) {
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE username = ? AND id != ?");
                $stmt->execute([$username, $user_id]);
                $count = $stmt->fetchColumn();
            } else {
                $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM users WHERE username = ? AND id != ?");
                $stmt->bind_param("si", $username, $user_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $row = $result->fetch_assoc();
                $count = $row['count'];
            }
            
            if ($count > 0) {
                $error_message = "用户名已被使用";
            } else {
                // 检查邮箱是否已被其他用户使用
                if ($pdo instanceof PDO) {
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE email = ? AND id != ?");
                    $stmt->execute([$email, $user_id]);
                    $count = $stmt->fetchColumn();
                } else {
                    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM users WHERE email = ? AND id != ?");
                    $stmt->bind_param("si", $email, $user_id);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    $row = $result->fetch_assoc();
                    $count = $row['count'];
                }
                
                if ($count > 0) {
                    $error_message = "邮箱已被其他用户使用";
                } else {
                    // 更新用户信息
                    if ($pdo instanceof PDO) {
                        $stmt = $pdo->prepare("UPDATE users SET username = ?, email = ? WHERE id = ?");
                        $stmt->execute([$username, $email, $user_id]);
                    } else {
                        $stmt = $pdo->prepare("UPDATE users SET username = ?, email = ? WHERE id = ?");
                        $stmt->bind_param("ssi", $username, $email, $user_id);
                        $stmt->execute();
                    }
                    
                    // 更新会话中的用户名
                    $_SESSION['username'] = $username;
                    
                    $success_message = "个人信息更新成功";
                    
                    // 重新获取用户信息
                    if ($pdo instanceof PDO) {
                        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                        $stmt->execute([$user_id]);
                        $user = $stmt->fetch(PDO::FETCH_ASSOC);
                    } else {
                        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                        $stmt->bind_param("i", $user_id);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        $user = $result->fetch_assoc();
                    }
                }
            }
        } catch (Exception $e) {
            $error_message = "更新个人信息失败: " . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>修改个人信息 - 举报系统</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            padding-top: 20px;
            padding-bottom: 20px;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            background-attachment: fixed;
            position: relative;
        }
        
        body::before {
            content: "";
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-image: url('https://cdn.pixabay.com/photo/2018/03/22/02/37/background-3249063_1280.png');
            background-size: cover;
            background-position: center;
            opacity: 0.1;
            z-index: -1;
        }
        
        .container {
            max-width: 700px;
            position: relative;
            z-index: 1;
        }
        
        .card {
            border-radius: 10px;
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
            border: none;
            background-color: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(5px);
        }
        
        .card-body {
            padding: 30px;
        }
        
        h1 {
            color: #0d6efd;
            font-weight: 600;
            text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.1);
        }
        
        .btn-back {
            color: #6c757d;
            text-decoration: none;
            transition: color 0.3s;
            display: inline-flex;
            align-items: center;
        }
        
        .btn-back:hover {
            color: #0d6efd;
        }
        
        .form-label {
            font-weight: 500;
        }
        
        .form-control:focus {
            border-color: #86b7fe;
            box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1>修改个人信息</h1>
            <a href="user_dashboard.php" class="btn-back">
                <i class="bi bi-arrow-left me-2"></i> 返回用户平台
            </a>
        </div>
        
        <?php if (!empty($success_message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="bi bi-check-circle-fill me-2"></i>
            <?php echo htmlspecialchars($success_message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($error_message)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="bi bi-exclamation-triangle-fill me-2"></i>
            <?php echo htmlspecialchars($error_message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-body">
                <form method="post" action="edit_profile.php">
                    <div class="mb-4 text-center">
                        <div class="display-1 text-primary">
                            <i class="bi bi-person-circle"></i>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="username" class="form-label">用户名</label>
                        <input type="text" class="form-control" id="username" name="username" value="<?php echo htmlspecialchars($user['username']); ?>" required>
                        <div class="form-text">用户名只能包含字母、数字和下划线，长度为3-20个字符</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="email" class="form-label">电子邮箱</label>
                        <input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($user['email']); ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">注册时间</label>
                        <input type="text" class="form-control" value="<?php echo date('Y-m-d H:i:s', strtotime($user['created_at'])); ?>" readonly>
                    </div>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-lg me-2"></i>保存修改
                        </button>
                        <a href="user_dashboard.php" class="btn btn-outline-secondary">
                            <i class="bi bi-x-lg me-2"></i>取消
                        </a>
                    </div>
                </form>
            </div>
        </div>
        
        <footer class="mt-5 text-center text-muted">
            <small>&copy; <?php echo date('Y'); ?> 举报系统 - 共建健康网络环境</small>
        </footer>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>