<?php
session_start();

// 检查是否已登录
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: login.php");
    exit;
}

require_once '../db_connect.php';

// 检查是否提供了举报ID
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: index.php");
    exit;
}

$report_id = (int)$_GET['id'];

// 获取举报详情
$stmt = $conn->prepare("SELECT * FROM reports WHERE id = ?");
$stmt->bind_param("i", $report_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows !== 1) {
    header("Location: index.php");
    exit;
}

$report = $result->fetch_assoc();

// 处理状态更新
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_status'])) {
    $new_status = $_POST['status'];
    
    // 验证状态值
    $allowed_statuses = ['pending', 'processing', 'resolved', 'rejected'];
    if (in_array($new_status, $allowed_statuses)) {
        $update_stmt = $conn->prepare("UPDATE reports SET status = ? WHERE id = ?");
        $update_stmt->bind_param("si", $new_status, $report_id);
        
        if ($update_stmt->execute()) {
            // 更新成功，刷新页面
            header("Location: view_report.php?id=$report_id&success=1");
            exit;
        }
        
        $update_stmt->close();
    }
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>举报详情 - 举报系统</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <style>
        body {
            padding-top: 20px;
            padding-bottom: 20px;
        }
        .container {
            max-width: 900px;
        }
        .badge.bg-pending { background-color: #6c757d; }
        .badge.bg-processing { background-color: #0d6efd; }
        .badge.bg-resolved { background-color: #198754; }
        .badge.bg-rejected { background-color: #dc3545; }
    </style>
</head>
<body>
    <div class="container">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1>举报详情</h1>
            <a href="index.php" class="btn btn-outline-secondary">返回列表</a>
        </div>
        
        <?php if(isset($_GET['success'])): ?>
        <div class="alert alert-success">
            状态更新成功
        </div>
        <?php endif; ?>
        
        <div class="card mb-4">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">举报 #<?php echo $report['id']; ?></h5>
                <?php
                $status_text = '';
                $status_class = '';
                switch ($report['status']) {
                    case 'pending':
                        $status_text = '待处理';
                        $status_class = 'bg-pending';
                        break;
                    case 'processing':
                        $status_text = '处理中';
                        $status_class = 'bg-processing';
                        break;
                    case 'resolved':
                        $status_text = '已解决';
                        $status_class = 'bg-resolved';
                        break;
                    case 'rejected':
                        $status_text = '已驳回';
                        $status_class = 'bg-rejected';
                        break;
                }
                ?>
                <span class="badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
            </div>
            <div class="card-body">
                <div class="row mb-3">
                    <div class="col-md-6">
                        <h6>举报人信息</h6>
                        <p><strong>姓名：</strong> <?php echo htmlspecialchars($report['reporter_name']); ?></p>
                        <p><strong>联系方式：</strong> <?php echo htmlspecialchars($report['reporter_contact']); ?></p>
                    </div>
                    <div class="col-md-6">
                        <h6>举报信息</h6>
                        <p><strong>类型：</strong> <?php echo htmlspecialchars($report['report_type']); ?></p>
                        <p><strong>提交时间：</strong> <?php echo date('Y-m-d H:i:s', strtotime($report['created_at'])); ?></p>
                    </div>
                </div>
                
                <div class="mb-3">
                    <h6>举报内容</h6>
                    <div class="p-3 bg-light rounded">
                        <?php echo nl2br(htmlspecialchars($report['report_content'])); ?>
                    </div>
                </div>
                
                <?php if (!empty($report['reported_url'])): ?>
                <div class="mb-3">
                    <h6>相关链接</h6>
                    <a href="<?php echo htmlspecialchars($report['reported_url']); ?>" target="_blank">
                        <?php echo htmlspecialchars($report['reported_url']); ?>
                    </a>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($report['evidence_file'])): ?>
                <div class="mb-3">
                    <h6>证据文件</h6>
                    <?php
                    $file_extension = pathinfo($report['evidence_file'], PATHINFO_EXTENSION);
                    $image_extensions = ['jpg', 'jpeg', 'png', 'gif'];
                    
                    if (in_array(strtolower($file_extension), $image_extensions)):
                    ?>
                        <div class="mb-2">
                            <img src="../<?php echo htmlspecialchars($report['evidence_file']); ?>" class="img-fluid" style="max-height: 300px;" alt="证据图片">
                        </div>
                    <?php endif; ?>
                    
                    <a href="../<?php echo htmlspecialchars($report['evidence_file']); ?>" class="btn btn-sm btn-outline-primary" target="_blank">
                        查看/下载文件
                    </a>
                </div>
                <?php endif; ?>
                
                <hr>
                
                <form method="post" class="mt-4">
                    <h6>更新状态</h6>
                    <div class="row g-3 align-items-center">
                        <div class="col-auto">
                            <select class="form-select" name="status">
                                <option value="pending" <?php echo $report['status'] === 'pending' ? 'selected' : ''; ?>>待处理</option>
                                <option value="processing" <?php echo $report['status'] === 'processing' ? 'selected' : ''; ?>>处理中</option>
                                <option value="resolved" <?php echo $report['status'] === 'resolved' ? 'selected' : ''; ?>>已解决</option>
                                <option value="rejected" <?php echo $report['status'] === 'rejected' ? 'selected' : ''; ?>>已驳回</option>
                            </select>
                        </div>
                        <div class="col-auto">
                            <button type="submit" name="update_status" class="btn btn-primary">更新状态</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>