<?php
session_start();

// 检查是否已登录
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: login.php");
    exit;
}

require_once '../db_connect.php';

// 确保数据库连接变量名正确
// 如果db_connect.php中使用的是$pdo，我们需要将其赋值给$conn
if (!isset($conn) && isset($pdo)) {
    $conn = $pdo;
}

// 如果仍然没有有效的数据库连接，显示错误
if (!isset($conn)) {
    die("数据库连接失败：db_connect.php 中没有定义 \$conn 变量");
}

// 获取筛选条件
$status_filter = isset($_GET['status']) ? $_GET['status'] : '';
$type_filter = isset($_GET['type']) ? $_GET['type'] : '';

// 构建查询
$query = "SELECT * FROM reports WHERE 1=1";
$params = [];
$types = "";

if (!empty($status_filter)) {
    $query .= " AND status = ?";
    $params[] = $status_filter;
    $types .= "s";
}

if (!empty($type_filter)) {
    $query .= " AND report_type = ?";
    $params[] = $type_filter;
    $types .= "s";
}

$query .= " ORDER BY created_at DESC";

// 准备和执行查询
$stmt = $conn->prepare($query);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();
$reports = $result->fetch_all(MYSQLI_ASSOC);

// 获取举报类型列表
$type_query = "SELECT DISTINCT report_type FROM reports ORDER BY report_type";
$type_result = $conn->query($type_query);
$report_types = [];
while ($row = $type_result->fetch_assoc()) {
    $report_types[] = $row['report_type'];
}

// 获取用户列表
$user_query = "SELECT id, username, email, created_at FROM users ORDER BY created_at DESC";
$user_result = $conn->query($user_query);
$users = [];
if ($user_result) {
    $users = $user_result->fetch_all(MYSQLI_ASSOC);
}

// 加载通知设置
$settings = [];
$config_file = '../config/notification_settings.php';
if (file_exists($config_file)) {
    try {
        $settings = require $config_file;
    } catch (Exception $e) {
        // 设置加载失败，使用空数组
        error_log('加载通知设置失败: ' . $e->getMessage());
    }
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>举报管理 - 举报系统</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        body {
            padding-top: 20px;
            padding-bottom: 20px;
        }
        .badge.bg-pending { background-color: #6c757d; }
        .badge.bg-processing { background-color: #0d6efd; }
        .badge.bg-resolved { background-color: #198754; }
        .badge.bg-rejected { background-color: #dc3545; }
        
        .notification-settings {
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 1000;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1>举报管理</h1>
            <div>
                <span>欢迎，<?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
                <button class="btn btn-outline-primary ms-2" data-bs-toggle="modal" data-bs-target="#notificationSettingsModal">
                    <i class="bi bi-bell"></i> 通知设置
                </button>
                <a href="logout.php" class="btn btn-outline-danger ms-2">退出登录</a>
            </div>
        </div>
        
        <!-- 添加导航标签页 -->
        <ul class="nav nav-tabs mb-4" id="adminTabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link active" id="reports-tab" data-bs-toggle="tab" data-bs-target="#reports-tab-pane" type="button" role="tab" aria-controls="reports-tab-pane" aria-selected="true">
                    <i class="bi bi-flag"></i> 举报管理
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="users-tab" data-bs-toggle="tab" data-bs-target="#users-tab-pane" type="button" role="tab" aria-controls="users-tab-pane" aria-selected="false">
                    <i class="bi bi-people"></i> 用户管理
                </button>
            </li>
        </ul>
        
        <div class="tab-content" id="adminTabsContent">
            <!-- 举报管理标签页 -->
            <div class="tab-pane fade show active" id="reports-tab-pane" role="tabpanel" aria-labelledby="reports-tab" tabindex="0">
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="get" class="row g-3">
                            <div class="col-md-5">
                                <label for="status" class="form-label">状态筛选</label>
                                <select class="form-select" id="status" name="status">
                                    <option value="">全部状态</option>
                                    <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>待处理</option>
                                    <option value="processing" <?php echo $status_filter === 'processing' ? 'selected' : ''; ?>>处理中</option>
                                    <option value="resolved" <?php echo $status_filter === 'resolved' ? 'selected' : ''; ?>>已解决</option>
                                    <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>已驳回</option>
                                </select>
                            </div>
                            <div class="col-md-5">
                                <label for="type" class="form-label">类型筛选</label>
                                <select class="form-select" id="type" name="type">
                                    <option value="">全部类型</option>
                                    <?php foreach ($report_types as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type); ?>" <?php echo $type_filter === $type ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($type); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-2 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary w-100">筛选</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-body">
                        <?php if (count($reports) > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>举报人</th>
                                        <th>类型</th>
                                        <th>内容摘要</th>
                                        <th>提交时间</th>
                                        <th>状态</th>
                                        <th>操作</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($reports as $report): ?>
                                    <tr>
                                        <td><?php echo $report['id']; ?></td>
                                        <td><?php echo htmlspecialchars($report['reporter_name']); ?></td>
                                        <td><?php echo htmlspecialchars($report['report_type']); ?></td>
                                        <td><?php echo htmlspecialchars(mb_substr($report['report_content'], 0, 30)) . (mb_strlen($report['report_content']) > 30 ? '...' : ''); ?></td>
                                        <td><?php echo date('Y-m-d H:i', strtotime($report['created_at'])); ?></td>
                                        <td>
                                            <?php
                                            $status_text = '';
                                            $status_class = '';
                                            switch ($report['status']) {
                                                case 'pending':
                                                    $status_text = '待处理';
                                                    $status_class = 'bg-pending';
                                                    break;
                                                case 'processing':
                                                    $status_text = '处理中';
                                                    $status_class = 'bg-processing';
                                                    break;
                                                case 'resolved':
                                                    $status_text = '已解决';
                                                    $status_class = 'bg-resolved';
                                                    break;
                                                case 'rejected':
                                                    $status_text = '已驳回';
                                                    $status_class = 'bg-rejected';
                                                    break;
                                            }
                                            ?>
                                            <span class="badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                                        </td>
                                        <td>
                                            <a href="view_report.php?id=<?php echo $report['id']; ?>" class="btn btn-sm btn-primary">查看</a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <div class="alert alert-info">
                            没有找到符合条件的举报信息
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- 用户管理标签页 -->
            <div class="tab-pane fade" id="users-tab-pane" role="tabpanel" aria-labelledby="users-tab" tabindex="0">
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h5 class="card-title mb-0">已注册用户列表</h5>
                            <div class="input-group" style="max-width: 300px;">
                                <input type="text" class="form-control" id="userSearchInput" placeholder="搜索用户...">
                                <button class="btn btn-outline-secondary" type="button" id="userSearchBtn">
                                    <i class="bi bi-search"></i>
                                </button>
                            </div>
                        </div>
                        
                        <?php if (count($users) > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover" id="usersTable">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>用户名</th>
                                        <th>邮箱</th>
                                        <th>注册时间</th>
                                        <th>操作</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($users as $user): ?>
                                    <tr>
                                        <td><?php echo $user['id']; ?></td>
                                        <td><?php echo htmlspecialchars($user['username']); ?></td>
                                        <td><?php echo htmlspecialchars($user['email']); ?></td>
                                        <td><?php echo date('Y-m-d H:i', strtotime($user['created_at'])); ?></td>
                                        <td>
                                            <button type="button" class="btn btn-sm btn-info view-user-btn" data-user-id="<?php echo $user['id']; ?>" data-bs-toggle="modal" data-bs-target="#viewUserModal">
                                                <i class="bi bi-eye"></i> 查看
                                            </button>
                                            <button type="button" class="btn btn-sm btn-danger delete-user-btn" data-user-id="<?php echo $user['id']; ?>" data-username="<?php echo htmlspecialchars($user['username']); ?>">
                                                <i class="bi bi-trash"></i> 删除
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <div class="alert alert-info">
                            暂无注册用户
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- 通知设置模态框 -->
        <div class="modal fade" id="notificationSettingsModal" tabindex="-1" aria-labelledby="notificationSettingsModalLabel" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="notificationSettingsModalLabel">通知设置</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <!-- 添加导航标签页 -->
                        <ul class="nav nav-tabs mb-3" id="notificationTabs" role="tablist">
                            <li class="nav-item" role="presentation">
                                <button class="nav-link active" id="email-tab" data-bs-toggle="tab" data-bs-target="#email-tab-pane" type="button" role="tab" aria-controls="email-tab-pane" aria-selected="true">
                                    <i class="bi bi-envelope"></i> 邮件通知
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="browser-tab" data-bs-toggle="tab" data-bs-target="#browser-tab-pane" type="button" role="tab" aria-controls="browser-tab-pane" aria-selected="false">
                                    <i class="bi bi-browser-chrome"></i> 浏览器通知
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="wechat-tab" data-bs-toggle="tab" data-bs-target="#wechat-tab-pane" type="button" role="tab" aria-controls="wechat-tab-pane" aria-selected="false">
                                    <i class="bi bi-chat-dots"></i> 企业微信通知
                                </button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link" id="general-tab" data-bs-toggle="tab" data-bs-target="#general-tab-pane" type="button" role="tab" aria-controls="general-tab-pane" aria-selected="false">
                                    <i class="bi bi-gear"></i> 通用设置
                                </button>
                            </li>
                        </ul>
                        
                        <form id="notificationSettingsForm">
                            <div class="tab-content" id="notificationTabsContent">
                                <!-- 邮件通知设置 -->
                                <div class="tab-pane fade show active" id="email-tab-pane" role="tabpanel" aria-labelledby="email-tab" tabindex="0">
                                    <div class="mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" id="enableEmailNotifications" name="enable_email" 
                                                <?php echo isset($settings['enable_email']) && $settings['enable_email'] ? 'checked' : ''; ?>>
                                            <label class="form-check-label" for="enableEmailNotifications">启用邮件通知</label>
                                        </div>
                                        <small class="text-muted">当有新举报提交时，系统将发送邮件通知</small>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="adminEmail" class="form-label">管理员邮箱</label>
                                        <input type="email" class="form-control" id="adminEmail" name="admin_email" 
                                            value="<?php echo isset($settings['admin_email']) ? htmlspecialchars($settings['admin_email']) : ''; ?>" 
                                            placeholder="接收通知的邮箱地址">
                                    </div>
                                    
                                    <!-- SMTP邮件配置 -->
                                    <div class="mb-3 border-top pt-3">
                                        <h6>SMTP邮件服务器配置</h6>
                                        <div class="form-check form-switch mb-2">
                                            <input class="form-check-input" type="checkbox" id="useSmtp" name="use_smtp" 
                                                <?php echo isset($settings['use_smtp']) && $settings['use_smtp'] ? 'checked' : ''; ?>>
                                            <label class="form-check-label" for="useSmtp">使用SMTP发送邮件</label>
                                        </div>
                                        <small class="text-muted d-block mb-3">启用后将使用SMTP服务器发送邮件，而不是PHP内置mail()函数</small>
                                        
                                        <div id="smtpSettings" class="<?php echo (!isset($settings['use_smtp']) || !$settings['use_smtp']) ? 'd-none' : ''; ?>">
                                            <div class="mb-2">
                                                <label for="smtpHost" class="form-label">SMTP服务器地址</label>
                                                <input type="text" class="form-control" id="smtpHost" name="smtp_host" 
                                                    value="<?php echo isset($settings['smtp_host']) ? htmlspecialchars($settings['smtp_host']) : ''; ?>" 
                                                    placeholder="例如: smtp.qq.com">
                                            </div>
                                            <div class="mb-2">
                                                <label for="smtpPort" class="form-label">SMTP端口</label>
                                                <input type="number" class="form-control" id="smtpPort" name="smtp_port" 
                                                    value="<?php echo isset($settings['smtp_port']) ? htmlspecialchars($settings['smtp_port']) : '465'; ?>" 
                                                    placeholder="465或587">
                                            </div>
                                            <div class="mb-2">
                                                <label for="smtpUsername" class="form-label">SMTP用户名</label>
                                                <input type="text" class="form-control" id="smtpUsername" name="smtp_username" 
                                                    value="<?php echo isset($settings['smtp_username']) ? htmlspecialchars($settings['smtp_username']) : ''; ?>" 
                                                    placeholder="通常是完整的邮箱地址">
                                            </div>
                                            <div class="mb-2">
                                                <label for="smtpPassword" class="form-label">SMTP密码/授权码</label>
                                                <input type="password" class="form-control" id="smtpPassword" name="smtp_password" 
                                                    value="<?php echo isset($settings['smtp_password']) ? htmlspecialchars($settings['smtp_password']) : ''; ?>" 
                                                    placeholder="邮箱密码或应用专用授权码">
                                            </div>
                                            <div class="mb-2">
                                                <label for="smtpSender" class="form-label">发件人邮箱</label>
                                                <input type="email" class="form-control" id="smtpSender" name="smtp_sender" 
                                                    value="<?php echo isset($settings['smtp_sender']) ? htmlspecialchars($settings['smtp_sender']) : ''; ?>" 
                                                    placeholder="显示的发件人邮箱地址">
                                            </div>
                                            <div class="mb-2">
                                                <label for="smtpSenderName" class="form-label">发件人名称</label>
                                                <input type="text" class="form-control" id="smtpSenderName" name="smtp_sender_name" 
                                                    value="<?php echo isset($settings['smtp_sender_name']) ? htmlspecialchars($settings['smtp_sender_name']) : '举报系统'; ?>" 
                                                    placeholder="显示的发件人名称">
                                            </div>
                                            <div class="mb-2">
                                                <label for="smtpSecure" class="form-label">加密方式</label>
                                                <select class="form-select" id="smtpSecure" name="smtp_secure">
                                                    <option value="ssl" <?php echo (isset($settings['smtp_secure']) && $settings['smtp_secure'] === 'ssl') ? 'selected' : ''; ?>>SSL</option>
                                                    <option value="tls" <?php echo (isset($settings['smtp_secure']) && $settings['smtp_secure'] === 'tls') ? 'selected' : ''; ?>>TLS</option>
                                                    <option value="none" <?php echo (isset($settings['smtp_secure']) && $settings['smtp_secure'] === 'none') ? 'selected' : ''; ?>>无加密</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- 测试邮件通知按钮 -->
                                    <div class="mb-3">
                                        <button type="button" class="btn btn-outline-info" id="testEmailNotification">
                                            <i class="bi bi-envelope"></i> 测试邮件通知
                                        </button>
                                        <small class="text-muted d-block mt-1">发送测试邮件，验证邮件配置是否正确</small>
                                    </div>
                                </div>
                                
                                <!-- 浏览器通知设置 -->
                                <div class="tab-pane fade" id="browser-tab-pane" role="tabpanel" aria-labelledby="browser-tab" tabindex="0">
                                    <div class="mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" id="enableBrowserNotifications" name="enable_browser" 
                                                <?php echo isset($settings['enable_browser']) && $settings['enable_browser'] ? 'checked' : ''; ?>>
                                            <label class="form-check-label" for="enableBrowserNotifications">启用浏览器通知</label>
                                        </div>
                                        <small class="text-muted">当有新举报提交时，系统将在浏览器中显示通知</small>
                                    </div>
                                    
                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle"></i> 浏览器通知需要您的浏览器支持并授予通知权限。
                                    </div>
                                    
                                    <div class="mb-3">
                                        <button type="button" class="btn btn-outline-info" id="testBrowserNotification">
                                            <i class="bi bi-browser-chrome"></i> 测试浏览器通知
                                        </button>
                                        <small class="text-muted d-block mt-1">发送测试通知，验证浏览器通知是否正常工作</small>
                                    </div>
                                </div>
                                
                                <!-- 企业微信通知设置 -->
                                <div class="tab-pane fade" id="wechat-tab-pane" role="tabpanel" aria-labelledby="wechat-tab" tabindex="0">
                                    <div class="mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" id="enableWechatWorkNotifications" name="enable_wechat_work" 
                                                <?php echo isset($settings['enable_wechat_work']) && $settings['enable_wechat_work'] ? 'checked' : ''; ?>>
                                            <label class="form-check-label" for="enableWechatWorkNotifications">启用企业微信机器人通知</label>
                                        </div>
                                        <small class="text-muted">当有新举报提交时，系统将通过企业微信机器人发送通知</small>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="wechatWorkWebhook" class="form-label">企业微信机器人Webhook地址</label>
                                        <input type="text" class="form-control" id="wechatWorkWebhook" name="wechat_work_webhook" 
                                            value="<?php echo isset($settings['wechat_work_webhook']) ? htmlspecialchars($settings['wechat_work_webhook']) : ''; ?>" 
                                            placeholder="https://qyapi.weixin.qq.com/cgi-bin/webhook/send?key=xxxxxxxx">
                                        <small class="text-muted">在企业微信群聊中添加机器人后获取的Webhook地址</small>
                                    </div>
                                    
                                    <!-- 测试通知按钮 -->
                                    <div class="mb-3">
                                        <button type="button" class="btn btn-outline-info" id="testWechatWorkNotification">
                                            <i class="bi bi-send"></i> 测试企业微信通知
                                        </button>
                                        <small class="text-muted d-block mt-1">发送测试消息到企业微信，验证配置是否正确</small>
                                    </div>
                                </div>
                                
                                <!-- 通用设置 -->
                                <div class="tab-pane fade" id="general-tab-pane" role="tabpanel" aria-labelledby="general-tab" tabindex="0">
                                    <div class="mb-3">
                                        <label class="form-label">通知频率</label>
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="notification_frequency" id="frequencyImmediate" value="immediate" 
                                                <?php echo (!isset($settings['notification_frequency']) || $settings['notification_frequency'] === 'immediate') ? 'checked' : ''; ?>>
                                            <label class="form-check-label" for="frequencyImmediate">
                                                立即通知
                                            </label>
                                            <small class="text-muted d-block">每当有新举报提交时立即发送通知</small>
                                        </div>
                                        <div class="form-check mt-2">
                                            <input class="form-check-input" type="radio" name="notification_frequency" id="frequencyDaily" value="daily" 
                                                <?php echo (isset($settings['notification_frequency']) && $settings['notification_frequency'] === 'daily') ? 'checked' : ''; ?>>
                                            <label class="form-check-label" for="frequencyDaily">
                                                每日摘要
                                            </label>
                                            <small class="text-muted d-block">每天汇总一次所有新举报，发送一条摘要通知</small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">取消</button>
                        <button type="button" class="btn btn-primary" id="saveNotificationSettings">保存设置</button>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- 查看用户详情模态框 -->
        <div class="modal fade" id="viewUserModal" tabindex="-1" aria-labelledby="viewUserModalLabel" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="viewUserModalLabel">用户详情</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <div class="text-center mb-3">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">加载中...</span>
                            </div>
                            <p>加载用户信息...</p>
                        </div>
                        <div id="userDetails" style="display: none;">
                            <!-- 用户详情将通过AJAX加载 -->
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">关闭</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- 现有的Bootstrap和其他脚本 -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- 添加通知设置表单的AJAX提交处理 -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // 通知设置保存功能
            document.getElementById('saveNotificationSettings').addEventListener('click', function() {
                const form = document.getElementById('notificationSettingsForm');
                const formData = new FormData(form);
                
                // 转换为对象
                const settings = {};
                formData.forEach((value, key) => {
                    if (key === 'enable_email' || key === 'enable_browser' || key === 'enable_wechat_work') {
                        settings[key] = true; // 复选框被选中时才会出现在FormData中
                    } else {
                        settings[key] = value;
                    }
                });
                
                // 处理未选中的复选框
                if (!formData.has('enable_email')) {
                    settings.enable_email = false;
                }
                if (!formData.has('enable_browser')) {
                    settings.enable_browser = false;
                }
                if (!formData.has('enable_wechat_work')) {
                    settings.enable_wechat_work = false;
                }
                
                // 发送AJAX请求保存设置
                fetch('save_notification_settings.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(settings)
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('通知设置已保存');
                        // 关闭模态框
                        const modal = bootstrap.Modal.getInstance(document.getElementById('notificationSettingsModal'));
                        modal.hide();
                    } else {
                        alert('保存设置失败: ' + data.message);
                    }
                })
                .catch(error => {
                    alert('保存请求失败: ' + error.message);
                });
            });
            
            // 测试邮件通知
            document.getElementById('testEmailNotification').addEventListener('click', function() {
                const email = document.getElementById('adminEmail').value.trim();
                
                if (!email) {
                    alert('请先填写管理员邮箱地址');
                    return;
                }
                
                // 禁用按钮，显示加载状态
                const button = this;
                const originalText = button.innerHTML;
                button.disabled = true;
                button.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> 发送中...';
                
                // 发送测试通知
                fetch('test_email_notification.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({ email: email })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('测试邮件发送成功！请检查邮箱是否收到邮件。');
                    } else {
                        alert('测试邮件发送失败: ' + data.message);
                    }
                })
                .catch(error => {
                    alert('发送请求失败: ' + error.message);
                })
                .finally(() => {
                    // 恢复按钮状态
                    button.disabled = false;
                    button.innerHTML = originalText;
                });
            });
            
            // SMTP设置显示控制
            document.getElementById('useSmtp').addEventListener('change', function() {
                const smtpSettings = document.getElementById('smtpSettings');
                if (this.checked) {
                    smtpSettings.classList.remove('d-none');
                } else {
                    smtpSettings.classList.add('d-none');
                }
            });
            
            // 测试浏览器通知
            document.getElementById('testBrowserNotification').addEventListener('click', function() {
                // 检查浏览器是否支持通知
                if (!("Notification" in window)) {
                    alert("您的浏览器不支持桌面通知");
                    return;
                }
                
                // 请求通知权限
                if (Notification.permission !== "granted") {
                    Notification.requestPermission().then(function(permission) {
                        if (permission === "granted") {
                            sendTestBrowserNotification();
                        } else {
                            alert("您需要允许通知权限才能接收浏览器通知");
                        }
                    });
                } else {
                    sendTestBrowserNotification();
                }
                
                function sendTestBrowserNotification() {
                    const notification = new Notification("举报系统测试通知", {
                        body: "这是一条测试通知，如果您看到此通知，说明浏览器通知功能正常。",
                        icon: "/favicon.ico"
                    });
                    
                    notification.onclick = function() {
                        window.focus();
                        notification.close();
                    };
                    
                    setTimeout(() => {
                        notification.close();
                    }, 5000);
                    
                    alert("测试通知已发送，请查看浏览器通知区域");
                }
            });
            
            // 测试企业微信通知
            document.getElementById('testWechatWorkNotification').addEventListener('click', function() {
                const webhookUrl = document.getElementById('wechatWorkWebhook').value.trim();
                    
                    if (!webhookUrl) {
                        alert('请先填写企业微信机器人Webhook地址');
                        return;
                    }
                    
                    // 禁用按钮，显示加载状态
                    const button = this;
                    const originalText = button.innerHTML;
                    button.disabled = true;
                    button.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> 发送中...';
                    
                    // 发送测试通知
                    fetch('test_wechat_work_notification.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({ webhook_url: webhookUrl })
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            alert('测试通知发送成功！请检查企业微信群是否收到消息。');
                        } else {
                            alert('测试通知发送失败: ' + data.message);
                        }
                    })
                    .catch(error => {
                        alert('发送请求失败: ' + error.message);
                    })
                    .finally(() => {
                        // 恢复按钮状态
                        button.disabled = false;
                        button.innerHTML = originalText;
                    });
                });
                
                // 用户搜索功能
                document.getElementById('userSearchBtn').addEventListener('click', searchUsers);
                
                document.getElementById('userSearchInput').addEventListener('keyup', function(e) {
                    if (e.key === 'Enter') {
                        searchUsers();
                    }
                });
                
                function searchUsers() {
                    const searchTerm = document.getElementById('userSearchInput').value.toLowerCase();
                    const table = document.getElementById('usersTable');
                    const rows = table.getElementsByTagName('tbody')[0].getElementsByTagName('tr');
                    
                    for (let i = 0; i < rows.length; i++) {
                        const username = rows[i].getElementsByTagName('td')[1].textContent.toLowerCase();
                        const email = rows[i].getElementsByTagName('td')[2].textContent.toLowerCase();
                        
                        if (username.includes(searchTerm) || email.includes(searchTerm)) {
                            rows[i].style.display = '';
                        } else {
                            rows[i].style.display = 'none';
                        }
                    }
                }
                
                // 查看用户详情
                const viewUserBtns = document.querySelectorAll('.view-user-btn');
                viewUserBtns.forEach(btn => {
                    btn.addEventListener('click', function() {
                        const userId = this.getAttribute('data-user-id');
                        const spinner = document.querySelector('#viewUserModal .spinner-border').parentElement;
                        const userDetails = document.getElementById('userDetails');
                        
                        spinner.style.display = 'block';
                        userDetails.style.display = 'none';
                        
                        // 发送AJAX请求获取用户详情
                        fetch(`get_user_details.php?id=${userId}`)
                            .then(response => response.json())
                            .then(data => {
                                spinner.style.display = 'none';
                                userDetails.style.display = 'block';
                                
                                if (data.success) {
                                    const user = data.user;
                                    let reportCount = data.report_count || 0;
                                    
                                    // 构建用户详情HTML
                                    let html = `
                                        <div class="text-center mb-4">
                                            <div class="display-1 text-primary">
                                                <i class="bi bi-person-circle"></i>
                                            </div>
                                            <h4>${htmlEscape(user.username)}</h4>
                                        </div>
                                        <div class="mb-3">
                                            <p><strong>用户ID:</strong> ${user.id}</p>
                                            <p><strong>邮箱:</strong> ${htmlEscape(user.email)}</p>
                                            <p><strong>注册时间:</strong> ${new Date(user.created_at).toLocaleString()}</p>
                                            <p><strong>提交举报数:</strong> ${reportCount}</p>
                                        </div>
                                    `;
                                    
                                    userDetails.innerHTML = html;
                                } else {
                                    userDetails.innerHTML = `<div class="alert alert-danger">${data.message}</div>`;
                                }
                            })
                            .catch(error => {
                                spinner.style.display = 'none';
                                userDetails.style.display = 'block';
                                userDetails.innerHTML = `<div class="alert alert-danger">获取用户信息失败: ${error.message}</div>`;
                            });
                    });
                });
                
                // 删除用户
                const deleteUserBtns = document.querySelectorAll('.delete-user-btn');
                deleteUserBtns.forEach(btn => {
                    btn.addEventListener('click', function() {
                        const userId = this.getAttribute('data-user-id');
                        const username = this.getAttribute('data-username');
                        
                        if (confirm(`确定要删除用户 "${username}" 吗？此操作不可撤销！`)) {
                            // 发送AJAX请求删除用户
                            fetch('delete_user.php', {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/x-www-form-urlencoded',
                                },
                                body: `id=${userId}`
                            })
                            .then(response => response.json())
                            .then(data => {
                                if (data.success) {
                                    // 删除成功，刷新页面
                                    alert(data.message);
                                    location.reload();
                                } else {
                                    alert(`删除失败: ${data.message}`);
                                }
                            })
                            .catch(error => {
                                alert(`删除请求失败: ${error.message}`);
                            });
                        }
                    });
                });
                
                // HTML转义函数
                function htmlEscape(str) {
                    return str
                        .replace(/&/g, '&amp;')
                        .replace(/"/g, '&quot;')
                        .replace(/'/g, '&#39;')
                        .replace(/</g, '&lt;')
                        .replace(/>/g, '&gt;');
                }
            });
            </script>
        </script>
</body>
</html>