// 数据持久化工具类
export class StorageManager {
  private static prefix = 'name-picker-';

  // 通用存储方法
  static set<T>(key: string, value: T): void {
    try {
      const serializedValue = JSON.stringify(value);
      localStorage.setItem(this.prefix + key, serializedValue);
    } catch (error) {
      console.error(`Failed to save ${key} to localStorage:`, error);
    }
  }

  // 通用读取方法
  static get<T>(key: string, defaultValue: T): T {
    try {
      const item = localStorage.getItem(this.prefix + key);
      if (item === null) return defaultValue;
      return JSON.parse(item) as T;
    } catch (error) {
      console.error(`Failed to load ${key} from localStorage:`, error);
      return defaultValue;
    }
  }

  // 删除数据
  static remove(key: string): void {
    try {
      localStorage.removeItem(this.prefix + key);
    } catch (error) {
      console.error(`Failed to remove ${key} from localStorage:`, error);
    }
  }

  // 清空所有数据
  static clear(): void {
    try {
      const keys = Object.keys(localStorage).filter(key => 
        key.startsWith(this.prefix)
      );
      keys.forEach(key => localStorage.removeItem(key));
    } catch (error) {
      console.error('Failed to clear localStorage:', error);
    }
  }

  // 获取存储大小
  static getStorageSize(): string {
    try {
      let total = 0;
      const keys = Object.keys(localStorage).filter(key => 
        key.startsWith(this.prefix)
      );
      keys.forEach(key => {
        const value = localStorage.getItem(key);
        if (value) {
          total += value.length;
        }
      });
      return (total / 1024).toFixed(2) + ' KB';
    } catch (error) {
      console.error('Failed to calculate storage size:', error);
      return '0 KB';
    }
  }
}

// 存储键名常量
export const STORAGE_KEYS = {
  STUDENTS: 'students',
  SETTINGS: 'settings',
  HISTORY: 'history',
  GROUPS: 'groups',
  SEATING: 'seating',
  ASSIGNMENTS: 'assignments',
  TIMER_SETTINGS: 'timer-settings',
  EXPORT_SETTINGS: 'export-settings',
  DICE_HISTORY: 'dice-history',
  WHEEL_HISTORY: 'wheel-history',
  CHALLENGE_HISTORY: 'challenge-history',
  BATTLE_HISTORY: 'battle-history',
  USER_PREFERENCES: 'user-preferences'
} as const;

// 数据类型定义
export interface StoredData {
  students: any[];
  settings: any;
  history: any[];
  groups: any[];
  seating: any;
  assignments: any[];
  timerSettings: any;
  exportSettings: any;
  diceHistory: any[];
  wheelHistory: any[];
  challengeHistory: any[];
  battleHistory: any[];
  userPreferences: any;
}

// 数据迁移和版本管理
export class DataMigration {
  private static currentVersion = '1.0.0';
  private static versionKey = 'data-version';

  static checkAndMigrate(): void {
    const currentVersion = StorageManager.get(this.versionKey, '0.0.0');
    
    if (currentVersion !== this.currentVersion) {
      console.log(`Migrating data from ${currentVersion} to ${this.currentVersion}`);
      this.performMigration(currentVersion);
      StorageManager.set(this.versionKey, this.currentVersion);
    }
  }

  private static performMigration(fromVersion: string): void {
    // 这里可以添加数据迁移逻辑
    // 例如：重命名字段、转换数据格式等
    console.log('Data migration completed');
  }
}