import React from 'react';
import { motion } from 'framer-motion';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { 
  Settings, 
  HelpCircle, 
  Search,
  Menu,
  User,
  ChevronDown
} from 'lucide-react';

interface TopBarProps {
  activeSection: string;
  studentsCount: number;
  sidebarCollapsed: boolean;
  onToggleSidebar: () => void;
}

const sectionTitles: Record<string, { title: string; subtitle: string }> = {
  dashboard: { title: '仪表盘', subtitle: '系统概览和快速操作' },
  students: { title: '学生管理', subtitle: '管理学生名单和信息' },
  picker: { title: '智能点名', subtitle: '随机选择学生进行互动' },
  attendance: { title: '考勤管理', subtitle: '学生出勤统计和管理' },
  classroom: { title: '课堂工具', subtitle: '分组、排座等课堂功能' },
  schedule: { title: '课程安排', subtitle: '课程时间和计划管理' },
  voting: { title: '课堂投票', subtitle: '实时投票和意见收集' },
  advanced: { title: '高级功能', subtitle: '权重设置和历史记录' },
  settings: { title: '系统设置', subtitle: '个性化配置和偏好设置' }
};

export function TopBar({ 
  activeSection, 
  studentsCount, 
  sidebarCollapsed, 
  onToggleSidebar 
}: TopBarProps) {
  const currentSection = sectionTitles[activeSection] || sectionTitles.dashboard;

  return (
    <motion.header
      initial={{ opacity: 0, y: -20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.3 }}
      className="sticky top-0 z-40 bg-white/80 backdrop-blur-xl border-b border-gray-200/50 shadow-sm"
    >
      <div className="flex items-center justify-between px-6 py-4">
        {/* 左侧：标题和面包屑 */}
        <div className="flex items-center space-x-4">
          {/* 移动端菜单按钮 */}
          <Button
            variant="ghost"
            size="sm"
            onClick={onToggleSidebar}
            className="lg:hidden p-2 hover:bg-gray-100 rounded-lg"
          >
            <Menu className="w-5 h-5 text-gray-600" />
          </Button>

          <div>
            <motion.h1 
              key={activeSection}
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.3 }}
              className="text-2xl font-bold text-gray-900"
            >
              {currentSection.title}
            </motion.h1>
            <motion.p 
              key={`${activeSection}-subtitle`}
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.3, delay: 0.1 }}
              className="text-sm text-gray-600 mt-1"
            >
              {currentSection.subtitle}
            </motion.p>
          </div>
        </div>

        {/* 占位空间 */}
        <div className="flex-1"></div>

        {/* 右侧：操作按钮 */}
        <div className="flex items-center space-x-3">
          {/* 学生数量显示 */}
          <div className="hidden sm:flex items-center space-x-2 px-3 py-2 bg-blue-50 rounded-xl">
            <User className="w-4 h-4 text-blue-600" />
            <span className="text-sm font-medium text-blue-700">
              {studentsCount} 名学生
            </span>
          </div>


          {/* 帮助按钮 */}
          <Button
            variant="ghost"
            size="sm"
            onClick={() => window.open('/help.html', '_blank')}
            className="p-2 hover:bg-gray-100 rounded-xl transition-colors"
            title="帮助文档"
          >
            <HelpCircle className="w-5 h-5 text-gray-600" />
          </Button>
        </div>
      </div>
    </motion.header>
  );
}