import React from 'react';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { 
  Users, 
  Dices, 
  GraduationCap, 
  Zap, 
  Wrench, 
  Download, 
  Settings,
  Home,
  Menu,
  X,
  ChevronRight,
  Calendar,
  Vote,
  HelpCircle
} from 'lucide-react';
import { motion, AnimatePresence } from 'framer-motion';

interface NavigationProps {
  activeSection: string;
  onSectionChange: (section: string) => void;
}

const navigationItems = [
  { id: 'namelist', label: '名单管理', icon: Users, count: 60, color: 'text-green-600', bgColor: 'bg-green-50', description: '管理学生名单' },
  { id: 'picker', label: '点名控制', icon: Dices, color: 'text-blue-600', bgColor: 'bg-blue-50', description: '智能随机选择学生' },
  { id: 'classroom', label: '课堂工具', icon: GraduationCap, color: 'text-orange-600', bgColor: 'bg-orange-50', description: '课堂互动功能' },
  { id: 'schedule', label: '课程安排', icon: Calendar, color: 'text-purple-600', bgColor: 'bg-purple-50', description: '课程时间管理' },
  { id: 'roleplay', label: '角色扮演', icon: Zap, color: 'text-red-600', bgColor: 'bg-red-50', description: '角色分配系统' },
  { id: 'voting', label: '课堂投票', icon: Vote, color: 'text-indigo-600', bgColor: 'bg-indigo-50', description: '实时投票功能' },
  { id: 'advanced', label: '高级功能', icon: Zap, color: 'text-purple-600', bgColor: 'bg-purple-50', description: '高级设置选项' },
  { id: 'utility', label: '实用工具', icon: Wrench, color: 'text-gray-600', bgColor: 'bg-gray-50', description: '实用小工具' },
  { id: 'export', label: '导出工具', icon: Download, color: 'text-teal-600', bgColor: 'bg-teal-50', description: '数据导出功能' },
  { id: 'settings', label: '系统设置', icon: Settings, color: 'text-gray-600', bgColor: 'bg-gray-50', description: '系统配置' }
];

export function Navigation({ activeSection, onSectionChange }: NavigationProps) {
  const [isCollapsed, setIsCollapsed] = React.useState(false);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = React.useState(false);

  return (
    <>
      {/* 桌面端侧边栏 */}
      <motion.div 
        className={`hidden lg:flex flex-col h-screen bg-white shadow-xl border-r border-gray-100 transition-all duration-300 ${
          isCollapsed ? 'w-20' : 'w-80'
        }`}
        initial={false}
        animate={{ width: isCollapsed ? 80 : 320 }}
      >
        {/* 头部 */}
        <div className="flex items-center justify-between p-6 bg-gradient-to-r from-blue-50 to-indigo-50 border-b border-gray-100">
          {!isCollapsed && (
            <motion.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              className="flex items-center space-x-3"
            >
              <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-2xl flex items-center justify-center shadow-lg">
                <Home className="w-7 h-7 text-white" />
              </div>
              <div>
                <h1 className="font-bold text-xl text-gray-900">柯猿课堂互动系统</h1>
                <p className="text-sm text-blue-600 font-medium">互动系统</p>
                <p className="text-xs text-gray-500">Keyuan Class Connect</p>
              </div>
            </motion.div>
          )}
          <div className="flex items-center space-x-2">
            {!isCollapsed && (
              <motion.div
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                exit={{ opacity: 0 }}
                className="flex items-center space-x-2"
              >
                <Button
                  variant="ghost"
                  size="sm"
                  className="p-3 hover:bg-white/50 rounded-xl transition-colors"
                  onClick={() => {
                    window.open('/help.html', '_blank');
                  }}
                  title="帮助文档"
                >
                  <HelpCircle className="w-5 h-5 text-blue-600" />
                </Button>
                
                <Button
                  variant="ghost"
                  size="sm"
                  className="p-3 hover:bg-white/50 rounded-xl transition-colors"
                  onClick={() => onSectionChange('settings')}
                  title="系统设置"
                >
                  <Settings className="w-5 h-5 text-blue-600" />
                </Button>
              </motion.div>
            )}
            <Button
              variant="ghost"
              size="sm"
              onClick={() => setIsCollapsed(!isCollapsed)}
              className="p-3 hover:bg-white/50 rounded-xl transition-colors"
            >
              <Menu className="w-5 h-5 text-blue-600" />
            </Button>
          </div>
        </div>

        {/* 导航项 */}
        <nav className="flex-1 p-4 space-y-2 overflow-y-auto">
          {navigationItems.map((item, index) => {
            const Icon = item.icon;
            const isActive = activeSection === item.id;
            
            return (
              <motion.div
                key={item.id}
                initial={{ opacity: 0, x: -20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.05 }}
                whileHover={{ scale: 1.02 }}
                whileTap={{ scale: 0.98 }}
              >
                <Button
                  variant="ghost"
                  className={`w-full justify-start h-16 rounded-2xl transition-all duration-200 border ${
                    isActive 
                      ? 'bg-blue-500 text-white shadow-lg border-blue-500 hover:bg-blue-600' 
                      : `text-gray-700 hover:${item.bgColor} hover:${item.color} border-transparent hover:border-gray-200 hover:shadow-sm`
                  } ${isCollapsed ? 'px-3' : 'px-5'}`}
                  onClick={() => onSectionChange(item.id)}
                >
                  <div className={`flex items-center ${isCollapsed ? 'justify-center' : 'justify-start'} w-full`}>
                    <div className={`p-2 rounded-xl ${isActive ? 'bg-white/20' : item.bgColor} transition-colors`}>
                      <Icon className={`w-6 h-6 ${isActive ? 'text-white' : item.color}`} />
                    </div>
                    {!isCollapsed && (
                      <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        exit={{ opacity: 0 }}
                        className="flex items-center justify-between w-full ml-4"
                      >
                        <div className="flex flex-col items-start">
                          <span className="font-semibold text-base">{item.label}</span>
                          {item.count && (
                            <span className={`text-sm ${isActive ? 'text-blue-100' : 'text-gray-500'}`}>
                              {item.count} 项
                            </span>
                          )}
                        </div>
                        <div className="flex items-center space-x-3">
                          {item.count && (
                            <Badge 
                              variant={isActive ? "secondary" : "outline"} 
                              className={`text-sm font-medium ${isActive ? 'bg-white text-blue-600' : 'border-gray-300'}`}
                            >
                              {item.count}
                            </Badge>
                          )}
                          {isActive && (
                            <motion.div
                              initial={{ scale: 0, rotate: -90 }}
                              animate={{ scale: 1, rotate: 0 }}
                              transition={{ type: "spring", stiffness: 500, damping: 30 }}
                            >
                              <ChevronRight className="w-5 h-5 text-white" />
                            </motion.div>
                          )}
                        </div>
                      </motion.div>
                    )}
                  </div>
                </Button>
              </motion.div>
            );
          })}
        </nav>

        {/* 底部信息 */}
        {!isCollapsed && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            className="p-6 bg-gradient-to-r from-gray-50 to-blue-50 border-t border-gray-100"
          >
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <span className="text-sm font-semibold text-gray-700">系统状态</span>
                <div className="flex items-center space-x-2">
                  <div className="w-3 h-3 bg-green-500 rounded-full animate-pulse shadow-sm"></div>
                  <span className="text-sm text-green-600 font-medium">系统正常</span>
                </div>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm text-gray-600">版本信息</span>
                <Badge variant="outline" className="text-sm border-blue-200 text-blue-600">v0.0.2</Badge>
              </div>
              <div className="text-xs text-gray-500 pt-2 border-t border-gray-200">
                © 2025 柯猿课堂互动系统 保留所有权利
              </div>
            </div>
          </motion.div>
        )}
      </motion.div>

      {/* 移动端顶部导航栏 */}
      <div className="lg:hidden bg-white shadow-lg border-b border-gray-100">
        <div className="flex items-center justify-between p-4">
          <div className="flex items-center space-x-3">
            <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-2xl flex items-center justify-center shadow-lg">
              <Home className="w-7 h-7 text-white" />
            </div>
            <div>
              <h1 className="font-bold text-xl text-gray-900">柯猿课堂互动系统</h1>
              <p className="text-sm text-blue-600 font-medium">Keyuan Class Connect v0.0.2</p>
            </div>
          </div>
          <div className="flex items-center space-x-2">
            <Button
              variant="ghost"
              size="sm"
              className="p-3 hover:bg-blue-50 rounded-xl transition-colors"
              onClick={() => {
                window.open('/help.html', '_blank');
              }}
              title="帮助文档"
            >
              <HelpCircle className="w-5 h-5 text-blue-600" />
            </Button>
            
            <Button
              variant="ghost"
              size="sm"
              className="p-3 hover:bg-blue-50 rounded-xl transition-colors"
              onClick={() => {
                onSectionChange('settings');
                setIsMobileMenuOpen(false);
              }}
              title="系统设置"
            >
              <Settings className="w-5 h-5 text-blue-600" />
            </Button>
            
            <Button
              variant="ghost"
              size="sm"
              onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
              className="p-3 hover:bg-blue-50 rounded-xl transition-colors"
            >
              {isMobileMenuOpen ? <X className="w-6 h-6 text-blue-600" /> : <Menu className="w-6 h-6 text-blue-600" />}
            </Button>
          </div>
        </div>

        {/* 移动端菜单 */}
        <AnimatePresence mode="wait">
          {isMobileMenuOpen && (
            <motion.div
              initial={{ height: 0, opacity: 0 }}
              animate={{ height: "auto", opacity: 1 }}
              exit={{ height: 0, opacity: 0 }}
              transition={{ duration: 0.2, ease: "easeInOut" }}
              className="border-t bg-white shadow-xl"
            >
              <nav className="p-4 space-y-3 max-h-96 overflow-y-auto">
                {navigationItems.map((item, index) => {
                  const Icon = item.icon;
                  const isActive = activeSection === item.id;
                  
                  return (
                    <motion.div
                      key={item.id}
                      initial={{ opacity: 0, y: -10 }}
                      animate={{ opacity: 1, y: 0 }}
                      exit={{ opacity: 0, y: -10 }}
                      transition={{ delay: index * 0.03, duration: 0.2 }}
                    >
                      <Button
                        variant="ghost"
                        className={`w-full justify-start h-16 rounded-2xl border transition-all ${
                          isActive 
                            ? 'bg-blue-500 text-white shadow-lg border-blue-500' 
                            : `text-gray-700 hover:${item.bgColor} hover:${item.color} border-transparent hover:border-gray-200 hover:shadow-sm`
                        }`}
                        onClick={() => {
                          onSectionChange(item.id);
                          setIsMobileMenuOpen(false);
                        }}
                      >
                        <div className={`p-2 rounded-xl ${isActive ? 'bg-white/20' : item.bgColor} transition-colors`}>
                          <Icon className={`w-6 h-6 ${isActive ? 'text-white' : item.color}`} />
                        </div>
                        <div className="flex items-center justify-between w-full ml-4">
                          <div className="flex flex-col items-start">
                            <span className="font-semibold text-base">{item.label}</span>
                            {item.count && (
                              <span className={`text-sm ${isActive ? 'text-blue-100' : 'text-gray-500'}`}>
                                {item.count} 项
                              </span>
                            )}
                          </div>
                          {item.count && (
                            <Badge 
                              variant={isActive ? "secondary" : "outline"} 
                              className={`text-sm font-medium ${isActive ? 'bg-white text-blue-600' : 'border-gray-300'}`}
                            >
                              {item.count}
                            </Badge>
                          )}
                        </div>
                      </Button>
                    </motion.div>
                  );
                })}
              </nav>
            </motion.div>
          )}
        </AnimatePresence>
      </div>

      {/* 移动端遮罩 */}
      <AnimatePresence mode="wait">
        {isMobileMenuOpen && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            transition={{ duration: 0.2 }}
            className="lg:hidden fixed inset-0 bg-black bg-opacity-25 z-40"
            onClick={() => setIsMobileMenuOpen(false)}
          />
        )}
      </AnimatePresence>
    </>
  );
}