import React, { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Badge } from './ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { Textarea } from './ui/textarea';
import { useToast } from './ui/use-toast';
import { 
  Gamepad2, 
  Dice1, 
  Dice2, 
  Dice3, 
  Dice4, 
  Dice5, 
  Dice6,
  Sparkles,
  Zap,
  Volume2,
  VolumeX,
  RotateCcw,
  Users,
  BookOpen,
  Plus,
  Trash2,
  CheckCircle,
  Calendar,
  User
} from 'lucide-react';
import type { Student, PickResult } from '../App';

interface ClassroomToolsProps {
  students: Student[];
  onPickResult: (result: PickResult) => void;
}

interface Assignment {
  id: string;
  subject: string;
  title: string;
  description: string;
  assignedTo: Student[];
  dueDate: string;
  status: 'pending' | 'completed';
  createdAt: Date;
}

interface GameSession {
  id: string;
  type: 'challenge' | 'team';
  participants: Student[];
  startTime: Date;
  status: 'active' | 'completed';
}

export function ClassroomTools({ students, onPickResult }: ClassroomToolsProps) {
  const [diceResult, setDiceResult] = useState<number>(1);
  const [isRolling, setIsRolling] = useState(false);
  const [wheelStudents, setWheelStudents] = useState<Student[]>([]);
  const [isSpinning, setIsSpinning] = useState(false);
  const [selectedStudent, setSelectedStudent] = useState<Student | null>(null);
  const [wheelRotation, setWheelRotation] = useState(0);
  
  // 从本地存储加载作业数据
  const [assignments, setAssignments] = useState<Assignment[]>(() => {
    try {
      const savedAssignments = localStorage.getItem('classroom-assignments');
      if (savedAssignments) {
        const parsed = JSON.parse(savedAssignments);
        // 恢复Date对象
        return parsed.map((assignment: any) => ({
          ...assignment,
          createdAt: new Date(assignment.createdAt)
        }));
      }
      return [];
    } catch (error) {
      console.error('Failed to load assignments from localStorage:', error);
      return [];
    }
  });
  
  const [newAssignment, setNewAssignment] = useState({
    subject: '',
    title: '',
    description: '',
    dueDate: '',
    assignedStudents: [] as string[]
  });
  
  const [soundEnabled, setSoundEnabled] = useState(true);
  const [gameSession, setGameSession] = useState<GameSession | null>(null);
  const [teamA, setTeamA] = useState<Student[]>([]);
  const [teamB, setTeamB] = useState<Student[]>([]);
  
  const { toast } = useToast();

  // 学科列表
  const subjects = [
    { id: 'chinese', name: '语文', icon: '📚', color: 'text-red-500' },
    { id: 'math', name: '数学', icon: '🔢', color: 'text-blue-500' },
    { id: 'english', name: '英语', icon: '🌍', color: 'text-green-500' },
    { id: 'science', name: '科学', icon: '🔬', color: 'text-purple-500' },
    { id: 'history', name: '历史', icon: '📜', color: 'text-yellow-600' },
    { id: 'geography', name: '地理', icon: '🌏', color: 'text-teal-500' },
    { id: 'physics', name: '物理', icon: '⚛️', color: 'text-indigo-500' },
    { id: 'chemistry', name: '化学', icon: '🧪', color: 'text-orange-500' },
    { id: 'biology', name: '生物', icon: '🌱', color: 'text-green-600' },
    { id: 'art', name: '美术', icon: '🎨', color: 'text-pink-500' },
    { id: 'music', name: '音乐', icon: '🎵', color: 'text-violet-500' },
    { id: 'pe', name: '体育', icon: '⚽', color: 'text-emerald-500' }
  ];

  // 初始化
  useEffect(() => {
    setWheelStudents(students);
  }, [students]);

  // 保存作业数据到本地存储
  useEffect(() => {
    try {
      localStorage.setItem('classroom-assignments', JSON.stringify(assignments));
    } catch (error) {
      console.error('Failed to save assignments to localStorage:', error);
    }
  }, [assignments]);

  // 播放音效
  const playSound = (frequency: number, duration: number = 200) => {
    if (!soundEnabled) return;
    
    try {
      const audioContext = new (window.AudioContext || (window as any).webkitAudioContext)();
      const oscillator = audioContext.createOscillator();
      const gainNode = audioContext.createGain();
      
      oscillator.connect(gainNode);
      gainNode.connect(audioContext.destination);
      
      oscillator.frequency.setValueAtTime(frequency, audioContext.currentTime);
      gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
      gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + duration / 1000);
      
      oscillator.start(audioContext.currentTime);
      oscillator.stop(audioContext.currentTime + duration / 1000);
    } catch (error) {
      console.log('Audio not supported');
    }
  };

  // 掷骰子功能
  const rollDice = () => {
    if (isRolling) return;
    
    setIsRolling(true);
    playSound(400, 100);
    
    let rollCount = 0;
    const rollInterval = setInterval(() => {
      setDiceResult(Math.floor(Math.random() * 6) + 1);
      rollCount++;
      
      if (rollCount >= 15) {
        clearInterval(rollInterval);
        const finalResult = Math.floor(Math.random() * 6) + 1;
        setDiceResult(finalResult);
        setIsRolling(false);
        playSound(800, 300);
        
        toast({
          title: "掷骰子结果",
          description: `🎲 点数: ${finalResult}`,
        });
      }
    }, 80);
  };

  // 幸运转盘功能
  const spinWheel = () => {
    if (isSpinning || wheelStudents.length === 0) return;
    
    setIsSpinning(true);
    setSelectedStudent(null);
    playSound(600, 100);
    
    const randomRotation = 1800 + Math.random() * 1800;
    setWheelRotation(prev => prev + randomRotation);
    
    setTimeout(() => {
      const finalStudent = wheelStudents[Math.floor(Math.random() * wheelStudents.length)];
      setSelectedStudent(finalStudent);
      setIsSpinning(false);
      playSound(1000, 500);
      
      const result: PickResult = {
        students: [finalStudent],
        timestamp: new Date(),
        mode: 'single'
      };
      onPickResult(result);
      
      toast({
        title: "🎉 幸运转盘",
        description: `恭喜 ${finalStudent.name} 被选中！`,
      });
    }, 3000);
  };

  // 获取骰子图标
  const getDiceIcon = (number: number) => {
    const icons = [Dice1, Dice2, Dice3, Dice4, Dice5, Dice6];
    const Icon = icons[number - 1];
    return <Icon className="h-16 w-16" />;
  };

  // 随机挑战模式
  const startChallenge = () => {
    if (students.length === 0) return;
    
    const challenges = [
      "背诵一首古诗词",
      "说出5个英语单词",
      "做10个俯卧撑",
      "唱一首歌曲",
      "讲一个有趣的故事",
      "画一个简笔画",
      "说出今天学到的知识点",
      "表演一个动作让大家猜",
      "用英语自我介绍",
      "说出5个成语"
    ];
    
    const randomStudent = students[Math.floor(Math.random() * students.length)];
    const randomChallenge = challenges[Math.floor(Math.random() * challenges.length)];
    
    setSelectedStudent(randomStudent);
    
    const result: PickResult = {
      students: [randomStudent],
      timestamp: new Date(),
      mode: 'single'
    };
    onPickResult(result);
    
    setGameSession({
      id: Date.now().toString(),
      type: 'challenge',
      participants: [randomStudent],
      startTime: new Date(),
      status: 'active'
    });
    
    playSound(600, 300);
    
    toast({
      title: "🎯 挑战模式",
      description: `${randomStudent.name}，请${randomChallenge}！`,
      duration: 8000,
    });
  };

  // 团队对战
  const startTeamBattle = () => {
    if (students.length < 2) {
      toast({
        title: "人数不足",
        description: "至少需要2名学生才能进行团队对战",
        variant: "destructive",
      });
      return;
    }
    
    const shuffled = [...students].sort(() => Math.random() - 0.5);
    const mid = Math.ceil(shuffled.length / 2);
    const newTeamA = shuffled.slice(0, mid);
    const newTeamB = shuffled.slice(mid);
    
    setTeamA(newTeamA);
    setTeamB(newTeamB);
    
    setGameSession({
      id: Date.now().toString(),
      type: 'team',
      participants: students,
      startTime: new Date(),
      status: 'active'
    });
    
    playSound(700, 400);
    
    toast({
      title: "⚔️ 团队对战开始",
      description: `红队 ${newTeamA.length} 人 VS 蓝队 ${newTeamB.length} 人`,
      duration: 5000,
    });
  };

  // 创建作业
  const createAssignment = () => {
    if (!newAssignment.subject || !newAssignment.title || !newAssignment.dueDate) {
      toast({
        title: "信息不完整",
        description: "请填写完整的作业信息",
        variant: "destructive",
      });
      return;
    }

    const assignedStudents = newAssignment.assignedStudents.length > 0 
      ? students.filter(s => newAssignment.assignedStudents.includes(s.id))
      : students;

    const assignment: Assignment = {
      id: Date.now().toString(),
      subject: newAssignment.subject,
      title: newAssignment.title,
      description: newAssignment.description,
      assignedTo: assignedStudents,
      dueDate: newAssignment.dueDate,
      status: 'pending',
      createdAt: new Date()
    };

    setAssignments(prev => [assignment, ...prev]);
    
    setNewAssignment({
      subject: '',
      title: '',
      description: '',
      dueDate: '',
      assignedStudents: []
    });

    toast({
      title: "📝 作业创建成功",
      description: `${assignment.title} 已分配给 ${assignedStudents.length} 名学生`,
    });
  };

  // 删除作业
  const deleteAssignment = (id: string) => {
    setAssignments(prev => prev.filter(a => a.id !== id));
    toast({
      title: "作业已删除",
      description: "作业已从列表中移除",
    });
  };

  // 标记作业完成
  const toggleAssignmentStatus = (id: string) => {
    setAssignments(prev => prev.map(a => 
      a.id === id 
        ? { ...a, status: a.status === 'pending' ? 'completed' : 'pending' }
        : a
    ));
  };

  // 获取学科信息
  const getSubjectInfo = (subjectId: string) => {
    return subjects.find(s => s.id === subjectId) || subjects[0];
  };

  return (
    <Card className="h-full">
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <Gamepad2 className="h-5 w-5" />
          <span>课堂互动</span>
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setSoundEnabled(!soundEnabled)}
            className="ml-auto"
          >
            {soundEnabled ? <Volume2 className="h-4 w-4" /> : <VolumeX className="h-4 w-4" />}
          </Button>
        </CardTitle>
      </CardHeader>
      
      <CardContent>
        <Tabs defaultValue="dice" className="w-full">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="dice">🎲 骰子</TabsTrigger>
            <TabsTrigger value="wheel">🎡 转盘</TabsTrigger>
            <TabsTrigger value="homework">📚 作业</TabsTrigger>
            <TabsTrigger value="games">🎮 游戏</TabsTrigger>
          </TabsList>

          {/* 掷骰子 */}
          <TabsContent value="dice" className="space-y-4">
            <div className="text-center space-y-4">
              <motion.div
                animate={isRolling ? { 
                  rotate: [0, 90, 180, 270, 360],
                  scale: [1, 1.1, 1, 1.1, 1]
                } : { rotate: 0, scale: 1 }}
                transition={{ 
                  duration: 0.1, 
                  repeat: isRolling ? Infinity : 0,
                  ease: "easeInOut"
                }}
                className="flex justify-center"
              >
                <div className={`
                  p-6 rounded-xl border-4 shadow-lg transition-all duration-300
                  ${isRolling ? 
                    'border-blue-500 bg-gradient-to-br from-blue-50 to-purple-50 dark:from-blue-900/20 dark:to-purple-900/20 shadow-blue-200' : 
                    'border-gray-300 bg-white dark:bg-gray-800 hover:shadow-md'}
                `}>
                  {getDiceIcon(diceResult)}
                </div>
              </motion.div>
              
              <motion.div 
                className="text-6xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent"
                animate={isRolling ? { scale: [1, 1.2, 1] } : { scale: 1 }}
                transition={{ duration: 0.3, repeat: isRolling ? Infinity : 0 }}
              >
                {diceResult}
              </motion.div>
              
              <Button
                onClick={rollDice}
                disabled={isRolling}
                className="w-full h-12 text-lg"
                size="lg"
              >
                {isRolling ? (
                  <>
                    <RotateCcw className="h-5 w-5 mr-2 animate-spin" />
                    掷骰子中...
                  </>
                ) : (
                  <>
                    <Dice1 className="h-5 w-5 mr-2" />
                    掷骰子
                  </>
                )}
              </Button>
              
              <div className="p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg text-sm text-gray-600 dark:text-gray-400">
                🎲 点击按钮掷骰子，可用于随机选择、游戏决策等场景
              </div>
            </div>
          </TabsContent>

          {/* 幸运转盘 */}
          <TabsContent value="wheel" className="space-y-4">
            <div className="text-center space-y-4">
              <div className="relative">
                <motion.div
                  animate={{ rotate: wheelRotation }}
                  transition={{ 
                    duration: isSpinning ? 3 : 0,
                    ease: isSpinning ? "easeOut" : "linear"
                  }}
                  className={`
                    w-40 h-40 mx-auto rounded-full border-8 flex items-center justify-center relative overflow-hidden
                    ${isSpinning ? 
                      'border-transparent bg-gradient-to-r from-pink-500 via-purple-500 via-blue-500 via-green-500 to-yellow-500' : 
                      'border-blue-500 bg-gradient-to-br from-blue-100 to-purple-100 dark:from-blue-900/20 dark:to-purple-900/20'}
                  `}
                  style={{
                    background: isSpinning ? 
                      'conic-gradient(from 0deg, #ff6b6b, #4ecdc4, #45b7d1, #96ceb4, #feca57, #ff9ff3, #54a0ff, #5f27cd)' :
                      undefined
                  }}
                >
                  <div className="absolute inset-2 rounded-full bg-white dark:bg-gray-800 flex items-center justify-center">
                    <Sparkles className={`h-16 w-16 ${isSpinning ? 'text-purple-600 animate-pulse' : 'text-blue-600'}`} />
                  </div>
                </motion.div>
                
                <div className="absolute top-0 left-1/2 transform -translate-x-1/2 -translate-y-2">
                  <div className="w-0 h-0 border-l-4 border-r-4 border-b-8 border-l-transparent border-r-transparent border-b-red-500"></div>
                </div>
                
                <AnimatePresence>
                  {selectedStudent && !isSpinning && (
                    <motion.div
                      initial={{ scale: 0, opacity: 0, y: 20 }}
                      animate={{ scale: 1, opacity: 1, y: 0 }}
                      exit={{ scale: 0, opacity: 0 }}
                      className="absolute -bottom-12 left-1/2 transform -translate-x-1/2"
                    >
                      <Badge className="text-xl px-6 py-3 bg-gradient-to-r from-yellow-400 to-orange-500 text-white shadow-lg">
                        🎉 {selectedStudent.name}
                      </Badge>
                    </motion.div>
                  )}
                </AnimatePresence>
              </div>
              
              <Button
                onClick={spinWheel}
                disabled={isSpinning || wheelStudents.length === 0}
                className="w-full h-12 text-lg"
                size="lg"
              >
                {isSpinning ? (
                  <>
                    <Sparkles className="h-5 w-5 mr-2 animate-spin" />
                    转盘旋转中...
                  </>
                ) : (
                  <>
                    <Sparkles className="h-5 w-5 mr-2" />
                    开始转盘
                  </>
                )}
              </Button>
              
              <div className="p-3 bg-purple-50 dark:bg-purple-900/20 rounded-lg text-sm text-gray-600 dark:text-gray-400">
                🎡 幸运转盘随机选择学生，增加课堂趣味性和参与度
              </div>
            </div>
          </TabsContent>

          {/* 各科作业 */}
          <TabsContent value="homework" className="space-y-4">
            <div className="space-y-4">
              {/* 创建作业表单 */}
              <div className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
                <h4 className="font-semibold mb-3 flex items-center">
                  <Plus className="h-4 w-4 mr-2 text-blue-600" />
                  创建新作业
                </h4>
                
                <div className="space-y-3">
                  <div>
                    <Label htmlFor="subject">学科</Label>
                    <select
                      id="subject"
                      value={newAssignment.subject}
                      onChange={(e) => setNewAssignment(prev => ({ ...prev, subject: e.target.value }))}
                      className="w-full mt-1 p-2 border rounded-md bg-white dark:bg-gray-800"
                    >
                      <option value="">选择学科</option>
                      {subjects.map(subject => (
                        <option key={subject.id} value={subject.id}>
                          {subject.icon} {subject.name}
                        </option>
                      ))}
                    </select>
                  </div>
                  
                  <div>
                    <Label htmlFor="title">作业标题</Label>
                    <Input
                      id="title"
                      value={newAssignment.title}
                      onChange={(e) => setNewAssignment(prev => ({ ...prev, title: e.target.value }))}
                      placeholder="输入作业标题"
                      className="mt-1"
                    />
                  </div>
                  
                  <div>
                    <Label htmlFor="description">作业描述</Label>
                    <Textarea
                      id="description"
                      value={newAssignment.description}
                      onChange={(e) => setNewAssignment(prev => ({ ...prev, description: e.target.value }))}
                      placeholder="输入作业详细描述"
                      className="mt-1"
                      rows={3}
                    />
                  </div>
                  
                  <div>
                    <Label htmlFor="dueDate">截止日期</Label>
                    <Input
                      id="dueDate"
                      type="date"
                      value={newAssignment.dueDate}
                      onChange={(e) => setNewAssignment(prev => ({ ...prev, dueDate: e.target.value }))}
                      className="mt-1"
                    />
                  </div>
                  
                  <Button onClick={createAssignment} className="w-full">
                    <Plus className="h-4 w-4 mr-2" />
                    创建作业
                  </Button>
                </div>
              </div>
              
              {/* 作业列表 */}
              <div className="space-y-3">
                <h4 className="font-semibold flex items-center">
                  <BookOpen className="h-4 w-4 mr-2 text-green-600" />
                  作业列表 ({assignments.length})
                </h4>
                
                <div className="max-h-64 overflow-y-auto space-y-2">
                  {assignments.length === 0 ? (
                    <div className="text-center py-8 text-gray-500 dark:text-gray-400">
                      <BookOpen className="h-12 w-12 mx-auto mb-2 opacity-50" />
                      <p>暂无作业</p>
                      <p className="text-sm">创建第一个作业吧！</p>
                    </div>
                  ) : (
                    assignments.map(assignment => {
                      const subjectInfo = getSubjectInfo(assignment.subject);
                      return (
                        <motion.div
                          key={assignment.id}
                          initial={{ opacity: 0, y: 20 }}
                          animate={{ opacity: 1, y: 0 }}
                          className={`
                            p-3 rounded-lg border transition-all hover:shadow-md
                            ${assignment.status === 'completed' 
                              ? 'bg-green-50 dark:bg-green-900/20 border-green-200 dark:border-green-800' 
                              : 'bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700'}
                          `}
                        >
                          <div className="flex items-start justify-between">
                            <div className="flex-1">
                              <div className="flex items-center space-x-2 mb-2">
                                <span className="text-lg">{subjectInfo.icon}</span>
                                <Badge variant="outline" className={subjectInfo.color}>
                                  {subjectInfo.name}
                                </Badge>
                                <Badge className={
                                  assignment.status === 'completed' 
                                    ? 'bg-green-500' 
                                    : 'bg-orange-500'
                                }>
                                  {assignment.status === 'completed' ? '已完成' : '进行中'}
                                </Badge>
                              </div>
                              
                              <h5 className="font-medium text-lg mb-1">{assignment.title}</h5>
                              
                              {assignment.description && (
                                <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                                  {assignment.description}
                                </p>
                              )}
                              
                              <div className="flex items-center space-x-4 text-xs text-gray-500 dark:text-gray-400">
                                <div className="flex items-center space-x-1">
                                  <Calendar className="h-3 w-3" />
                                  <span>截止: {assignment.dueDate}</span>
                                </div>
                                <div className="flex items-center space-x-1">
                                  <User className="h-3 w-3" />
                                  <span>{assignment.assignedTo.length} 人</span>
                                </div>
                              </div>
                            </div>
                            
                            <div className="flex items-center space-x-2 ml-4">
                              <Button
                                size="sm"
                                variant="ghost"
                                onClick={() => toggleAssignmentStatus(assignment.id)}
                                className={
                                  assignment.status === 'completed' 
                                    ? 'text-green-600 hover:text-green-700' 
                                    : 'text-blue-600 hover:text-blue-700'
                                }
                              >
                                <CheckCircle className="h-4 w-4" />
                              </Button>
                              <Button
                                size="sm"
                                variant="ghost"
                                onClick={() => deleteAssignment(assignment.id)}
                                className="text-red-600 hover:text-red-700"
                              >
                                <Trash2 className="h-4 w-4" />
                              </Button>
                            </div>
                          </div>
                        </motion.div>
                      );
                    })
                  )}
                </div>
              </div>
            </div>
          </TabsContent>

          {/* 互动游戏 */}
          <TabsContent value="games" className="space-y-4">
            <div className="space-y-4">
              {/* 当前游戏状态 */}
              {gameSession && gameSession.status === 'active' && (
                <div className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
                  <div className="flex items-center justify-between mb-2">
                    <h4 className="font-semibold text-blue-800 dark:text-blue-200">
                      🎮 游戏进行中
                    </h4>
                    <Badge className="bg-blue-500">
                      {gameSession.type === 'challenge' ? '挑战模式' : '团队对战'}
                    </Badge>
                  </div>
                  
                  {gameSession.type === 'team' && (
                    <div className="space-y-3">
                      <div className="grid grid-cols-2 gap-4">
                        <div className="p-3 bg-red-50 dark:bg-red-900/20 rounded-lg">
                          <h5 className="font-medium text-red-800 dark:text-red-200 mb-2">🔴 红队</h5>
                          <div className="space-y-1">
                            {teamA.map(student => (
                              <div key={student.id} className="text-sm">{student.name}</div>
                            ))}
                          </div>
                        </div>
                        <div className="p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                          <h5 className="font-medium text-blue-800 dark:text-blue-200 mb-2">🔵 蓝队</h5>
                          <div className="space-y-1">
                            {teamB.map(student => (
                              <div key={student.id} className="text-sm">{student.name}</div>
                            ))}
                          </div>
                        </div>
                      </div>
                    </div>
                  )}
                </div>
              )}
              
              <div className="grid grid-cols-1 gap-3">
                <Button onClick={startChallenge} className="w-full h-12" size="lg">
                  <Zap className="h-5 w-5 mr-2" />
                  随机挑战
                </Button>
                
                <Button
                  onClick={startTeamBattle}
                  variant="outline"
                  className="w-full h-12"
                  size="lg"
                >
                  <Users className="h-5 w-5 mr-2" />
                  团队对战
                </Button>
              </div>
              
              <div className="p-4 bg-gradient-to-r from-blue-50 to-purple-50 dark:from-blue-900/20 dark:to-purple-900/20 rounded-lg text-sm">
                <h4 className="font-semibold mb-3 flex items-center">
                  <Zap className="h-4 w-4 mr-2 text-blue-600" />
                  游戏说明
                </h4>
                <ul className="space-y-2 text-xs text-gray-600 dark:text-gray-400">
                  <li className="flex items-start space-x-2">
                    <span className="text-blue-500">•</span>
                    <span><strong>随机挑战:</strong> 为学生分配有趣的任务，完成后可获得成就感</span>
                  </li>
                  <li className="flex items-start space-x-2">
                    <span className="text-red-500">•</span>
                    <span><strong>团队对战:</strong> 将班级分为红蓝两队进行竞赛活动</span>
                  </li>
                </ul>
              </div>
            </div>
          </TabsContent>
        </Tabs>
      </CardContent>
    </Card>
  );
}
