import React, { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { Button } from './ui/button';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Progress } from './ui/progress';
import { useToast } from './ui/use-toast';
import { 
  UserCheck, 
  UserX, 
  Clock, 
  Calendar,
  Users,
  CheckCircle,
  XCircle,
  AlertCircle,
  Download,
  Upload,
  Search,
  Filter,
  BarChart3,
  TrendingUp,
  TrendingDown
} from 'lucide-react';
import { Student } from '../App';

interface AttendanceRecord {
  id: string;
  studentId: string;
  date: string;
  status: 'present' | 'absent' | 'late' | 'excused';
  timestamp: Date;
  note?: string;
}

interface AttendanceStats {
  totalStudents: number;
  presentCount: number;
  absentCount: number;
  lateCount: number;
  excusedCount: number;
  attendanceRate: number;
}

interface AttendanceProps {
  students: Student[];
}

export function Attendance({ students }: AttendanceProps) {
  const { toast } = useToast();
  const [attendanceRecords, setAttendanceRecords] = useState<AttendanceRecord[]>([]);
  const [selectedDate, setSelectedDate] = useState(new Date().toISOString().split('T')[0]);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterStatus, setFilterStatus] = useState<string>('all');
  const [isQuickAttendance, setIsQuickAttendance] = useState(false);

  // 从localStorage加载考勤记录
  useEffect(() => {
    const savedRecords = localStorage.getItem('attendance-records');
    if (savedRecords) {
      try {
        const parsedRecords = JSON.parse(savedRecords);
        const recordsWithDates = parsedRecords.map((record: any) => ({
          ...record,
          timestamp: new Date(record.timestamp)
        }));
        setAttendanceRecords(recordsWithDates);
      } catch (error) {
        console.error('Failed to load attendance records:', error);
      }
    }
  }, []);

  // 保存考勤记录到localStorage
  useEffect(() => {
    localStorage.setItem('attendance-records', JSON.stringify(attendanceRecords));
  }, [attendanceRecords]);

  // 获取今日考勤统计
  const getTodayStats = (): AttendanceStats => {
    const today = new Date().toISOString().split('T')[0];
    const todayRecords = attendanceRecords.filter(record => record.date === today);
    
    const stats = {
      totalStudents: students.length,
      presentCount: 0,
      absentCount: 0,
      lateCount: 0,
      excusedCount: 0,
      attendanceRate: 0
    };

    todayRecords.forEach(record => {
      switch (record.status) {
        case 'present':
          stats.presentCount++;
          break;
        case 'absent':
          stats.absentCount++;
          break;
        case 'late':
          stats.lateCount++;
          break;
        case 'excused':
          stats.excusedCount++;
          break;
      }
    });

    // 计算出勤率（出席+迟到）/ 总人数
    const attendedCount = stats.presentCount + stats.lateCount;
    stats.attendanceRate = stats.totalStudents > 0 ? (attendedCount / stats.totalStudents) * 100 : 0;

    return stats;
  };

  // 获取指定日期的考勤记录
  const getDateRecords = (date: string) => {
    return attendanceRecords.filter(record => record.date === date);
  };

  // 获取学生在指定日期的考勤状态
  const getStudentStatus = (studentId: string, date: string) => {
    const record = attendanceRecords.find(r => r.studentId === studentId && r.date === date);
    return record?.status || 'unmarked';
  };

  // 标记考勤
  const markAttendance = (studentId: string, status: 'present' | 'absent' | 'late' | 'excused', note?: string) => {
    const existingRecordIndex = attendanceRecords.findIndex(
      r => r.studentId === studentId && r.date === selectedDate
    );

    const newRecord: AttendanceRecord = {
      id: existingRecordIndex >= 0 ? attendanceRecords[existingRecordIndex].id : Date.now().toString(),
      studentId,
      date: selectedDate,
      status,
      timestamp: new Date(),
      note
    };

    if (existingRecordIndex >= 0) {
      // 更新现有记录
      setAttendanceRecords(prev => 
        prev.map((record, index) => index === existingRecordIndex ? newRecord : record)
      );
    } else {
      // 添加新记录
      setAttendanceRecords(prev => [...prev, newRecord]);
    }

    const student = students.find(s => s.id === studentId);
    const statusText = {
      present: '出席',
      absent: '缺席',
      late: '迟到',
      excused: '请假'
    }[status];

    toast({
      title: "考勤已更新",
      description: `${student?.name} - ${statusText}`
    });
  };

  // 批量标记出席
  const markAllPresent = () => {
    const today = new Date().toISOString().split('T')[0];
    students.forEach(student => {
      const existingRecord = attendanceRecords.find(
        r => r.studentId === student.id && r.date === today
      );
      if (!existingRecord) {
        markAttendance(student.id, 'present');
      }
    });
    toast({
      title: "批量操作完成",
      description: "所有未标记的学生已标记为出席"
    });
  };

  // 快速考勤模式
  const startQuickAttendance = () => {
    setIsQuickAttendance(true);
    setSelectedDate(new Date().toISOString().split('T')[0]);
    toast({
      title: "快速考勤模式",
      description: "点击学生姓名快速标记出席状态"
    });
  };

  // 导出考勤数据
  const exportAttendance = () => {
    const exportData = {
      exportDate: new Date().toISOString(),
      students: students.map(student => ({
        id: student.id,
        name: student.name,
        records: attendanceRecords.filter(r => r.studentId === student.id)
      })),
      summary: getTodayStats()
    };

    const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `考勤记录-${new Date().toLocaleDateString().replace(/\//g, '-')}.json`;
    link.click();
    URL.revokeObjectURL(url);

    toast({
      title: "导出成功",
      description: "考勤数据已导出"
    });
  };

  // 过滤学生列表
  const filteredStudents = students.filter(student => {
    const matchesSearch = student.name.toLowerCase().includes(searchTerm.toLowerCase());
    if (!matchesSearch) return false;

    if (filterStatus === 'all') return true;
    
    const status = getStudentStatus(student.id, selectedDate);
    return status === filterStatus || (filterStatus === 'unmarked' && status === 'unmarked');
  });

  const todayStats = getTodayStats();

  return (
    <div className="space-y-6">
      {/* 考勤统计卡片 */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card className="bg-gradient-to-br from-green-50 to-emerald-50 border-green-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-green-600">出席人数</p>
                <p className="text-2xl font-bold text-green-700">{todayStats.presentCount}</p>
              </div>
              <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                <UserCheck className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-red-50 to-rose-50 border-red-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-red-600">缺席人数</p>
                <p className="text-2xl font-bold text-red-700">{todayStats.absentCount}</p>
              </div>
              <div className="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center">
                <UserX className="w-6 h-6 text-red-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-yellow-50 to-amber-50 border-yellow-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-yellow-600">迟到人数</p>
                <p className="text-2xl font-bold text-yellow-700">{todayStats.lateCount}</p>
              </div>
              <div className="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center">
                <Clock className="w-6 h-6 text-yellow-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-blue-50 to-indigo-50 border-blue-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-blue-600">出勤率</p>
                <p className="text-2xl font-bold text-blue-700">{todayStats.attendanceRate.toFixed(1)}%</p>
              </div>
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                <BarChart3 className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* 操作控制区 */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="w-5 h-5 text-blue-600" />
            考勤管理
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap items-center gap-4 mb-4">
            <div className="flex items-center gap-2">
              <Label>日期:</Label>
              <Input
                type="date"
                value={selectedDate}
                onChange={(e) => setSelectedDate(e.target.value)}
                className="w-40"
              />
            </div>

            <div className="flex items-center gap-2">
              <Search className="w-4 h-4 text-gray-400" />
              <Input
                placeholder="搜索学生..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-40"
              />
            </div>

            <select
              value={filterStatus}
              onChange={(e) => setFilterStatus(e.target.value)}
              className="px-3 py-2 border border-gray-200 rounded-md text-sm"
            >
              <option value="all">全部状态</option>
              <option value="present">出席</option>
              <option value="absent">缺席</option>
              <option value="late">迟到</option>
              <option value="excused">请假</option>
              <option value="unmarked">未标记</option>
            </select>

            <div className="ml-auto flex gap-2">
              <Button variant="outline" onClick={startQuickAttendance}>
                <Clock className="w-4 h-4 mr-2" />
                快速考勤
              </Button>
              <Button variant="outline" onClick={markAllPresent}>
                <CheckCircle className="w-4 h-4 mr-2" />
                全部出席
              </Button>
              <Button variant="outline" onClick={exportAttendance}>
                <Download className="w-4 h-4 mr-2" />
                导出数据
              </Button>
            </div>
          </div>

          {/* 出勤率进度条 */}
          <div className="mb-4">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium">今日出勤率</span>
              <span className="text-sm text-gray-600">{todayStats.attendanceRate.toFixed(1)}%</span>
            </div>
            <Progress value={todayStats.attendanceRate} className="h-2" />
          </div>
        </CardContent>
      </Card>

      {/* 学生考勤列表 */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <Users className="w-5 h-5 text-purple-600" />
              学生考勤列表
            </div>
            <Badge variant="secondary">
              {filteredStudents.length} / {students.length} 名学生
            </Badge>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid gap-3">
            {filteredStudents.map((student) => {
              const status = getStudentStatus(student.id, selectedDate);
              const statusConfig = {
                present: { color: 'bg-green-100 text-green-700 border-green-200', icon: CheckCircle, text: '出席' },
                absent: { color: 'bg-red-100 text-red-700 border-red-200', icon: XCircle, text: '缺席' },
                late: { color: 'bg-yellow-100 text-yellow-700 border-yellow-200', icon: Clock, text: '迟到' },
                excused: { color: 'bg-blue-100 text-blue-700 border-blue-200', icon: AlertCircle, text: '请假' },
                unmarked: { color: 'bg-gray-100 text-gray-700 border-gray-200', icon: Users, text: '未标记' }
              };

              const config = statusConfig[status as keyof typeof statusConfig] || statusConfig.unmarked;
              const StatusIcon = config.icon;

              return (
                <motion.div
                  key={student.id}
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  className={`p-4 rounded-lg border-2 ${config.color} transition-all hover:shadow-md`}
                >
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white font-semibold">
                        {student.name.charAt(0)}
                      </div>
                      <div>
                        <h3 className="font-semibold">{student.name}</h3>
                        <div className="flex items-center gap-1 mt-1">
                          <StatusIcon className="w-4 h-4" />
                          <span className="text-sm">{config.text}</span>
                        </div>
                      </div>
                    </div>

                    <div className="flex items-center gap-2">
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => markAttendance(student.id, 'present')}
                        className="text-green-600 hover:bg-green-50"
                      >
                        出席
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => markAttendance(student.id, 'late')}
                        className="text-yellow-600 hover:bg-yellow-50"
                      >
                        迟到
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => markAttendance(student.id, 'absent')}
                        className="text-red-600 hover:bg-red-50"
                      >
                        缺席
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => markAttendance(student.id, 'excused')}
                        className="text-blue-600 hover:bg-blue-50"
                      >
                        请假
                      </Button>
                    </div>
                  </div>
                </motion.div>
              );
            })}
          </div>

          {filteredStudents.length === 0 && (
            <div className="text-center py-12">
              <Users className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">没有找到学生</h3>
              <p className="text-gray-600">请检查搜索条件或添加学生</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}