import React, { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Badge } from './ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { Textarea } from './ui/textarea';
import { Switch } from './ui/switch';
import { useToast } from './ui/use-toast';
import { StorageManager, STORAGE_KEYS } from '../utils/storage';
import { 
  Target, 
  Users, 
  Clock, 
  Trophy,
  Zap,
  BookOpen,
  CheckCircle,
  XCircle,
  RotateCcw,
  Save,
  Upload,
  Star,
  Award,
  Timer,
  Bell
} from 'lucide-react';
import type { Student, PickResult } from '../App';

interface AdvancedToolsProps {
  students: Student[];
  history: PickResult[];
  onPickResult: (result: PickResult) => void;
}

interface Task {
  id: string;
  title: string;
  description: string;
  assignedTo: Student[];
  status: 'pending' | 'completed' | 'failed';
  createdAt: Date;
  completedAt?: Date;
}

interface Quiz {
  id: string;
  question: string;
  options: string[];
  correctAnswer: number;
  difficulty: 'easy' | 'medium' | 'hard';
}

interface StudentScore {
  studentId: string;
  studentName: string;
  score: number;
  correctAnswers: number;
  totalQuestions: number;
  lastActivity: Date;
}

export function AdvancedTools({ students, history, onPickResult }: AdvancedToolsProps) {
  // 从localStorage加载初始状态
  const [tasks, setTasks] = useState<Task[]>(() => 
    StorageManager.get('tasks', [])
  );
  const [newTaskTitle, setNewTaskTitle] = useState('');
  const [newTaskDescription, setNewTaskDescription] = useState('');
  const [quizzes, setQuizzes] = useState<Quiz[]>(() => 
    StorageManager.get('quizzes', [])
  );
  const [currentQuiz, setCurrentQuiz] = useState<Quiz | null>(null);
  const [studentScores, setStudentScores] = useState<StudentScore[]>(() => 
    StorageManager.get('student-scores', [])
  );
  const [selectedStudents, setSelectedStudents] = useState<string[]>([]);
  const [isQuizMode, setIsQuizMode] = useState(false);
  const [timeRemaining, setTimeRemaining] = useState(() => 
    StorageManager.get('timer-remaining', 0)
  );
  const [timerActive, setTimerActive] = useState(false);
  const [attendanceList, setAttendanceList] = useState<{[key: string]: boolean}>(() => 
    StorageManager.get('attendance-list', {})
  );
  const { toast } = useToast();

  // 自动保存状态到localStorage
  useEffect(() => {
    StorageManager.set('tasks', tasks);
  }, [tasks]);

  useEffect(() => {
    StorageManager.set('quizzes', quizzes);
  }, [quizzes]);

  useEffect(() => {
    StorageManager.set('student-scores', studentScores);
  }, [studentScores]);

  useEffect(() => {
    StorageManager.set('timer-remaining', timeRemaining);
  }, [timeRemaining]);

  useEffect(() => {
    StorageManager.set('attendance-list', attendanceList);
  }, [attendanceList]);

  // 组件加载时恢复数据提示
  useEffect(() => {
    const savedTasks = StorageManager.get('tasks', []);
    const savedScores = StorageManager.get('student-scores', []);
    const savedAttendance = StorageManager.get('attendance-list', {});
    
    if (savedTasks.length > 0 || savedScores.length > 0 || Object.keys(savedAttendance).length > 0) {
      toast({
        title: "高级工具数据已恢复",
        description: `已恢复 ${savedTasks.length} 个任务和学生积分数据`,
      });
    }
  }, [toast]);

  // 初始化学生分数
  useEffect(() => {
    const scores = students.map(student => ({
      studentId: student.id,
      studentName: student.name,
      score: 0,
      correctAnswers: 0,
      totalQuestions: 0,
      lastActivity: new Date()
    }));
    setStudentScores(scores);
    
    // 初始化考勤
    const attendance: {[key: string]: boolean} = {};
    students.forEach(student => {
      attendance[student.id] = false;
    });
    setAttendanceList(attendance);
  }, [students]);

  // 计时器功能
  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (timerActive && timeRemaining > 0) {
      interval = setInterval(() => {
        setTimeRemaining(prev => {
          if (prev <= 1) {
            setTimerActive(false);
            toast({
              title: "时间到！",
              description: "计时器已结束",
            });
            return 0;
          }
          return prev - 1;
        });
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [timerActive, timeRemaining, toast]);

  // 任务分配功能
  const assignTask = () => {
    if (!newTaskTitle.trim()) {
      toast({
        title: "请输入任务标题",
        variant: "destructive",
      });
      return;
    }

    if (selectedStudents.length === 0) {
      toast({
        title: "请选择学生",
        description: "至少选择一名学生来分配任务",
        variant: "destructive",
      });
      return;
    }

    const assignedStudents = students.filter(s => selectedStudents.includes(s.id));
    const newTask: Task = {
      id: `task-${Date.now()}`,
      title: newTaskTitle,
      description: newTaskDescription,
      assignedTo: assignedStudents,
      status: 'pending',
      createdAt: new Date()
    };

    setTasks(prev => [newTask, ...prev]);
    setNewTaskTitle('');
    setNewTaskDescription('');
    setSelectedStudents([]);

    toast({
      title: "任务分配成功",
      description: `已为 ${assignedStudents.length} 名学生分配任务`,
    });
  };

  // 更新任务状态
  const updateTaskStatus = (taskId: string, status: Task['status']) => {
    setTasks(prev => prev.map(task => 
      task.id === taskId 
        ? { ...task, status, completedAt: status !== 'pending' ? new Date() : undefined }
        : task
    ));
  };

  // 随机提问功能
  const startRandomQuiz = () => {
    if (students.length === 0) {
      toast({
        title: "无法开始提问",
        description: "请先导入学生名单",
        variant: "destructive",
      });
      return;
    }

    const randomStudent = students[Math.floor(Math.random() * students.length)];
    const result: PickResult = {
      students: [randomStudent],
      timestamp: new Date(),
      mode: 'single'
    };
    
    onPickResult(result);
    setIsQuizMode(true);
    
    toast({
      title: "随机提问",
      description: `请 ${randomStudent.name} 回答问题`,
    });
  };

  // 答题评分
  const scoreAnswer = (studentName: string, isCorrect: boolean) => {
    setStudentScores(prev => prev.map(score => 
      score.studentName === studentName
        ? {
            ...score,
            score: score.score + (isCorrect ? 10 : 0),
            correctAnswers: score.correctAnswers + (isCorrect ? 1 : 0),
            totalQuestions: score.totalQuestions + 1,
            lastActivity: new Date()
          }
        : score
    ));

    toast({
      title: isCorrect ? "回答正确！" : "回答错误",
      description: `${studentName} ${isCorrect ? '获得 10 分' : '未得分'}`,
      variant: isCorrect ? "default" : "destructive",
    });

    setIsQuizMode(false);
  };

  // 考勤打卡
  const toggleAttendance = (studentId: string) => {
    setAttendanceList(prev => ({
      ...prev,
      [studentId]: !prev[studentId]
    }));
  };

  // 批量考勤
  const markAllPresent = () => {
    const newAttendance: {[key: string]: boolean} = {};
    students.forEach(student => {
      newAttendance[student.id] = true;
    });
    setAttendanceList(newAttendance);
    toast({
      title: "批量签到完成",
      description: "所有学生已标记为出席",
    });
  };

  // 启动计时器
  const startTimer = (minutes: number) => {
    setTimeRemaining(minutes * 60);
    setTimerActive(true);
    toast({
      title: "计时器已启动",
      description: `倒计时 ${minutes} 分钟`,
    });
  };

  // 格式化时间显示
  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
  };

  // 获取出席率
  const getAttendanceRate = () => {
    const presentCount = Object.values(attendanceList).filter(Boolean).length;
    return students.length > 0 ? (presentCount / students.length * 100).toFixed(1) : '0';
  };

  return (
    <Card className="h-full">
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <Zap className="h-5 w-5" />
          <span>高级工具</span>
        </CardTitle>
      </CardHeader>
      
      <CardContent>
        <Tabs defaultValue="tasks" className="w-full">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="tasks">任务</TabsTrigger>
            <TabsTrigger value="quiz">提问</TabsTrigger>
            <TabsTrigger value="attendance">考勤</TabsTrigger>
            <TabsTrigger value="timer">计时</TabsTrigger>
          </TabsList>

          {/* 任务分配 */}
          <TabsContent value="tasks" className="space-y-4">
            <div className="space-y-3">
              <div>
                <Label htmlFor="task-title">任务标题</Label>
                <Input
                  id="task-title"
                  value={newTaskTitle}
                  onChange={(e) => setNewTaskTitle(e.target.value)}
                  placeholder="输入任务标题..."
                />
              </div>
              
              <div>
                <Label htmlFor="task-description">任务描述</Label>
                <Textarea
                  id="task-description"
                  value={newTaskDescription}
                  onChange={(e) => setNewTaskDescription(e.target.value)}
                  placeholder="输入任务详细描述..."
                  rows={3}
                />
              </div>

              <div>
                <Label>选择学生</Label>
                <div className="flex flex-wrap gap-1 mt-2 max-h-32 overflow-y-auto">
                  {students.map(student => (
                    <Badge
                      key={student.id}
                      variant={selectedStudents.includes(student.id) ? "default" : "outline"}
                      className="cursor-pointer"
                      onClick={() => {
                        setSelectedStudents(prev => 
                          prev.includes(student.id)
                            ? prev.filter(id => id !== student.id)
                            : [...prev, student.id]
                        );
                      }}
                    >
                      {student.name}
                    </Badge>
                  ))}
                </div>
              </div>

              <Button onClick={assignTask} className="w-full">
                <Target className="h-4 w-4 mr-2" />
                分配任务
              </Button>

              {/* 任务列表 */}
              <div className="space-y-2 max-h-48 overflow-y-auto">
                <AnimatePresence>
                  {tasks.map(task => (
                    <motion.div
                      key={task.id}
                      initial={{ opacity: 0, y: 10 }}
                      animate={{ opacity: 1, y: 0 }}
                      exit={{ opacity: 0, y: -10 }}
                      className="p-3 bg-gray-50 dark:bg-gray-800 rounded-md"
                    >
                      <div className="flex items-center justify-between mb-2">
                        <h4 className="font-medium text-sm">{task.title}</h4>
                        <Badge 
                          variant={
                            task.status === 'completed' ? 'default' :
                            task.status === 'failed' ? 'destructive' : 'secondary'
                          }
                        >
                          {task.status === 'completed' ? '已完成' :
                           task.status === 'failed' ? '未完成' : '进行中'}
                        </Badge>
                      </div>
                      
                      {task.description && (
                        <p className="text-xs text-gray-600 dark:text-gray-400 mb-2">
                          {task.description}
                        </p>
                      )}
                      
                      <div className="flex items-center justify-between">
                        <div className="flex flex-wrap gap-1">
                          {task.assignedTo.map(student => (
                            <Badge key={student.id} variant="outline" className="text-xs">
                              {student.name}
                            </Badge>
                          ))}
                        </div>
                        
                        {task.status === 'pending' && (
                          <div className="flex space-x-1">
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => updateTaskStatus(task.id, 'completed')}
                              className="h-6 px-2"
                            >
                              <CheckCircle className="h-3 w-3" />
                            </Button>
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => updateTaskStatus(task.id, 'failed')}
                              className="h-6 px-2"
                            >
                              <XCircle className="h-3 w-3" />
                            </Button>
                          </div>
                        )}
                      </div>
                    </motion.div>
                  ))}
                </AnimatePresence>
              </div>
            </div>
          </TabsContent>

          {/* 随机提问和评分 */}
          <TabsContent value="quiz" className="space-y-4">
            <div className="space-y-3">
              <Button onClick={startRandomQuiz} className="w-full" disabled={isQuizMode}>
                <BookOpen className="h-4 w-4 mr-2" />
                {isQuizMode ? '提问进行中...' : '随机提问'}
              </Button>

              {isQuizMode && history.length > 0 && (
                <motion.div
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg text-center"
                >
                  <h3 className="text-lg font-bold text-blue-700 dark:text-blue-300 mb-2">
                    请 {history[0].students[0].name} 回答问题
                  </h3>
                  <div className="flex justify-center space-x-2">
                    <Button
                      onClick={() => scoreAnswer(history[0].students[0].name, true)}
                      className="bg-green-500 hover:bg-green-600"
                    >
                      <CheckCircle className="h-4 w-4 mr-1" />
                      正确 (+10分)
                    </Button>
                    <Button
                      onClick={() => scoreAnswer(history[0].students[0].name, false)}
                      variant="destructive"
                    >
                      <XCircle className="h-4 w-4 mr-1" />
                      错误 (0分)
                    </Button>
                  </div>
                </motion.div>
              )}

              {/* 学生积分排行榜 */}
              <div className="space-y-2">
                <h4 className="text-sm font-medium flex items-center">
                  <Trophy className="h-4 w-4 mr-1" />
                  积分排行榜
                </h4>
                <div className="max-h-48 overflow-y-auto space-y-1">
                  {studentScores
                    .sort((a, b) => b.score - a.score)
                    .map((score, index) => (
                      <div
                        key={score.studentId}
                        className={`
                          flex items-center justify-between p-2 rounded text-sm
                          ${index === 0 ? 'bg-yellow-100 dark:bg-yellow-900/20' :
                            index === 1 ? 'bg-gray-100 dark:bg-gray-800' :
                            index === 2 ? 'bg-orange-100 dark:bg-orange-900/20' :
                            'bg-gray-50 dark:bg-gray-800/50'}
                        `}
                      >
                        <div className="flex items-center space-x-2">
                          <span className="font-bold w-6">#{index + 1}</span>
                          {index < 3 && (
                            <Award className={`h-4 w-4 ${
                              index === 0 ? 'text-yellow-500' :
                              index === 1 ? 'text-gray-500' :
                              'text-orange-500'
                            }`} />
                          )}
                          <span>{score.studentName}</span>
                        </div>
                        <div className="text-right">
                          <div className="font-bold">{score.score}分</div>
                          <div className="text-xs text-gray-500">
                            {score.totalQuestions > 0 
                              ? `${score.correctAnswers}/${score.totalQuestions}`
                              : '未答题'
                            }
                          </div>
                        </div>
                      </div>
                    ))}
                </div>
              </div>
            </div>
          </TabsContent>

          {/* 考勤管理 */}
          <TabsContent value="attendance" className="space-y-4">
            <div className="space-y-3">
              <div className="flex items-center justify-between">
                <div className="text-sm">
                  <span className="font-medium">出席率: </span>
                  <span className="text-green-600 font-bold">{getAttendanceRate()}%</span>
                  <span className="text-gray-500 ml-2">
                    ({Object.values(attendanceList).filter(Boolean).length}/{students.length})
                  </span>
                </div>
                <Button onClick={markAllPresent} size="sm">
                  <CheckCircle className="h-4 w-4 mr-1" />
                  全部签到
                </Button>
              </div>

              <div className="grid grid-cols-1 gap-2 max-h-64 overflow-y-auto">
                {students.map(student => (
                  <div
                    key={student.id}
                    className={`
                      flex items-center justify-between p-2 rounded border
                      ${attendanceList[student.id] 
                        ? 'bg-green-50 dark:bg-green-900/20 border-green-200 dark:border-green-800' 
                        : 'bg-gray-50 dark:bg-gray-800 border-gray-200 dark:border-gray-700'
                      }
                    `}
                  >
                    <span className="text-sm">{student.name}</span>
                    <div className="flex items-center space-x-2">
                      <Badge 
                        variant={attendanceList[student.id] ? "default" : "secondary"}
                        className="text-xs"
                      >
                        {attendanceList[student.id] ? '出席' : '缺席'}
                      </Badge>
                      <Switch
                        checked={attendanceList[student.id]}
                        onCheckedChange={() => toggleAttendance(student.id)}
                      />
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </TabsContent>

          {/* 计时器工具 */}
          <TabsContent value="timer" className="space-y-4">
            <div className="space-y-3">
              <div className="text-center">
                <div className={`
                  text-6xl font-mono font-bold mb-4
                  ${timeRemaining <= 60 ? 'text-red-500' : 'text-blue-600'}
                `}>
                  {formatTime(timeRemaining)}
                </div>
                
                {timerActive && (
                  <motion.div
                    animate={{ scale: [1, 1.1, 1] }}
                    transition={{ duration: 1, repeat: Infinity }}
                  >
                    <Bell className="h-8 w-8 mx-auto text-blue-500" />
                  </motion.div>
                )}
              </div>

              <div className="grid grid-cols-3 gap-2">
                <Button
                  onClick={() => startTimer(5)}
                  disabled={timerActive}
                  variant="outline"
                >
                  5分钟
                </Button>
                <Button
                  onClick={() => startTimer(10)}
                  disabled={timerActive}
                  variant="outline"
                >
                  10分钟
                </Button>
                <Button
                  onClick={() => startTimer(15)}
                  disabled={timerActive}
                  variant="outline"
                >
                  15分钟
                </Button>
              </div>

              <div className="flex space-x-2">
                {!timerActive ? (
                  <Button
                    onClick={() => {
                      if (timeRemaining > 0) {
                        setTimerActive(true);
                      } else {
                        startTimer(5);
                      }
                    }}
                    className="flex-1"
                  >
                    <Timer className="h-4 w-4 mr-2" />
                    开始
                  </Button>
                ) : (
                  <Button
                    onClick={() => setTimerActive(false)}
                    variant="destructive"
                    className="flex-1"
                  >
                    暂停
                  </Button>
                )}
                
                <Button
                  onClick={() => {
                    setTimerActive(false);
                    setTimeRemaining(0);
                  }}
                  variant="outline"
                >
                  <RotateCcw className="h-4 w-4" />
                </Button>
              </div>

              {timeRemaining === 0 && !timerActive && (
                <div className="text-center p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                  <CheckCircle className="h-8 w-8 mx-auto text-green-500 mb-2" />
                  <p className="text-green-700 dark:text-green-300">计时结束！</p>
                </div>
              )}
            </div>
          </TabsContent>
        </Tabs>
      </CardContent>
    </Card>
  );
}